package ru.yandex.direct.web.entity.excel.service;

import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.direct.common.mds.MdsHolder;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.excel.processing.exception.ExcelValidationException;
import ru.yandex.inside.mds.MdsFileKey;
import ru.yandex.misc.io.InputStreamSource;

import static ru.yandex.direct.excel.processing.validation.defects.ExcelDefectIds.FILE_NOT_EXIST_IN_MDS;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;
import static ru.yandex.direct.web.entity.excel.model.internalad.InternalAdImportRequest.EXCEL_FILE_KEY;

@ParametersAreNonnullByDefault
public class ExcelWebUtil {

    private static final Logger logger = LoggerFactory.getLogger(ExcelWebUtil.class);
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd-HHmmss");

    static String sanitizeCampaignName(String campaignName) {
        return campaignName
                .replaceAll("[^\\w\\dА-Яа-я]", "_")
                .replaceAll("_+", "_");
    }

    static String generateMdsPath(ClientId clientId, Long campaignId, String campaignName) {
        String time = DATE_TIME_FORMATTER.format(LocalDateTime.now());
        String sanitizedCampaignName = sanitizeCampaignName(campaignName);
        return String.format("internal-ad-export/%d/%s/%d-%s.xlsx",
                clientId.asLong(), time, campaignId, sanitizedCampaignName);
    }

    static InputStream getInputStream(MultipartFile excelFile) {
        try {
            return excelFile.getInputStream();
        } catch (IOException e) {
            logger.error("got IOException when getInputStream of excelFile", e);
            throw new RuntimeException(e);
        }
    }

    static InputStream downloadExcelFileFromMds(MdsHolder mdsHolder, MdsFileKey fileKey) {
        InputStreamSource excelFile = mdsHolder.download(fileKey);

        if (!excelFile.exists()) {
            logger.warn("file {} not exist in MDS", fileKey.toString());
            throw ExcelValidationException.create(FILE_NOT_EXIST_IN_MDS, path(field(EXCEL_FILE_KEY)));
        }

        try {
            return excelFile.getInput();
        } catch (IOException e) {
            logger.error("got IOException when getInputStream of excelFile which downloaded from MDS", e);
            throw new RuntimeException(e);
        }
    }

}
