package ru.yandex.direct.web.entity.excel.service;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.direct.common.mds.MdsHolder;
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.model.BannerImageFromPool;
import ru.yandex.direct.core.entity.image.service.ImageService;
import ru.yandex.direct.core.entity.internalads.model.ReadOnlyDirectTemplateResource;
import ru.yandex.direct.core.entity.internalads.service.TemplateResourceService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.validation.ValidationUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.excel.processing.model.CampaignInfoWithWorkbook;
import ru.yandex.direct.excel.processing.model.ObjectType;
import ru.yandex.direct.excel.processing.model.internalad.ExcelFetchedData;
import ru.yandex.direct.excel.processing.model.internalad.ExcelImportResult;
import ru.yandex.direct.excel.processing.model.internalad.ExcelSheetFetchedData;
import ru.yandex.direct.excel.processing.model.internalad.InternalBannerRepresentation;
import ru.yandex.direct.excel.processing.service.internalad.InternalAdExcelImportService;
import ru.yandex.direct.excel.processing.service.internalad.InternalAdExcelService;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.excel.model.ExcelFileKey;
import ru.yandex.direct.web.entity.excel.model.UploadedImageInfo;
import ru.yandex.direct.web.entity.excel.model.internalad.AdGroupInfo;
import ru.yandex.direct.web.entity.excel.model.internalad.AdInfo;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdExportRequest;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdImportInfo;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdImportMode;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdImportRequest;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdTemplateExportRequest;
import ru.yandex.inside.mds.MdsFileKey;
import ru.yandex.inside.mds.MdsPostResponse;
import ru.yandex.inside.mds.MdsUrlUtils;
import ru.yandex.misc.io.ByteArrayInputStreamSource;

import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.web.entity.excel.service.ExcelWebConverter.getAdGroupsForAdd;
import static ru.yandex.direct.web.entity.excel.service.ExcelWebConverter.getAdGroupsInfo;
import static ru.yandex.direct.web.entity.excel.service.ExcelWebConverter.getTemplateVariableImageHashes;
import static ru.yandex.direct.web.entity.excel.service.ExcelWebConverter.importModeToObjectTypes;
import static ru.yandex.direct.web.entity.excel.service.ExcelWebConverter.requestToInternalAdExportParameters;
import static ru.yandex.direct.web.entity.excel.service.ExcelWebConverter.toAdsInfo;
import static ru.yandex.direct.web.entity.excel.service.ExcelWebConverter.toBytes;
import static ru.yandex.direct.web.entity.excel.service.ExcelWebConverter.toMdsFileKey;
import static ru.yandex.direct.web.entity.excel.service.ExcelWebUtil.downloadExcelFileFromMds;
import static ru.yandex.direct.web.entity.excel.service.ExcelWebUtil.generateMdsPath;
import static ru.yandex.direct.web.entity.excel.service.ExcelWebUtil.getInputStream;

@Service
@ParametersAreNonnullByDefault
public class InternalAdExcelWebService {

    private static final Logger logger = LoggerFactory.getLogger(InternalAdExcelWebService.class);

    private static final Duration EXPIRE_DURATION = Duration.standardHours(6);
    private static final String UNKNOWN_IMAGE_HASH = "UNKNOWN_IMAGE_HASH";

    private final InternalAdExcelService internalAdExcelService;
    private final InternalAdExcelImportService internalAdExcelImportService;
    private final MdsHolder mdsHolder;
    private final TemplateResourceService templateResourceService;
    private final CampaignService campaignService;
    private final ImageService imageService;
    private final UserService userService;

    @Autowired
    public InternalAdExcelWebService(MdsHolder mdsHolder, InternalAdExcelService internalAdExcelService,
                                     InternalAdExcelImportService internalAdExcelImportService,
                                     TemplateResourceService templateResourceService,
                                     CampaignService campaignService,
                                     ImageService imageService,
                                     UserService userService) {
        this.internalAdExcelService = internalAdExcelService;
        this.internalAdExcelImportService = internalAdExcelImportService;
        this.mdsHolder = mdsHolder;
        this.templateResourceService = templateResourceService;
        this.campaignService = campaignService;
        this.imageService = imageService;
        this.userService = userService;
    }

    /**
     * Возвращает урл на MDS, где лежит excel-файл с внутренней рекламой
     */
    public Result<String> exportInternal(ClientId clientId, InternalAdExportRequest request) {
        ValidationResult<InternalAdExportRequest, Defect> vr =
                InternalAdValidationService.validateExportRequest(request);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }

        CampaignInfoWithWorkbook campaignInfoWithWorkbook;
        try (TraceProfile profile = Trace.current().profile("internal_ad_export:create_workbook")) {
            campaignInfoWithWorkbook =
                    internalAdExcelService.createWorkbook(requestToInternalAdExportParameters(clientId, request));
        }

        String url = saveToMds(clientId, campaignInfoWithWorkbook);
        return Result.successful(url);
    }

    public Result<String> exportInternalTemplate(ClientId clientId, InternalAdTemplateExportRequest request) {
        ValidationResult<InternalAdTemplateExportRequest, Defect> vr =
                InternalAdValidationService.validateTemplateExportRequest(request);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }

        CampaignInfoWithWorkbook campaignInfoWithWorkbook;
        try (TraceProfile profile = Trace.current().profile("internal_ad_template_export:create_workbook")) {
            campaignInfoWithWorkbook = internalAdExcelService.createWorkbookForEmptyCampaign(
                    clientId, request.getCampaignId(), request.getPlaceId(), request.getTemplateId());
        }

        String url = saveToMds(clientId, campaignInfoWithWorkbook);
        return Result.successful(url);
    }

    /**
     * Возвращает необходимые параметры групп и объявлений для фронта и ключ загруженного файла в MDS
     * Получает на вход excel файл. Читает и валидирует содержимое. Если нет ошибок при валидации,
     * то сохраняет файл в MDS, чтобы фронт мог не присылать файл заново для второго шага импорта
     */
    public Result<InternalAdImportInfo> getDataFromExcelFileAndUploadFileToMds(ClientId clientId,
                                                                               InternalAdImportMode importMode,
                                                                               MultipartFile excelFile) {
        ValidationResult<MultipartFile, Defect> vr = InternalAdValidationService.validateImportFile(excelFile);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }

        Set<ObjectType> objectTypesForRead = importModeToObjectTypes(importMode);
        Result<ExcelFetchedData> dataFromExcelFile = internalAdExcelService
                .getAndValidateDataFromExcelFile(clientId, getInputStream(excelFile), objectTypesForRead);
        if (ValidationUtils.hasValidationIssues(dataFromExcelFile)) {
            return Result.broken(dataFromExcelFile.getValidationResult());
        }

        InternalAdImportInfo importInfo = getImportInfo(clientId, objectTypesForRead, dataFromExcelFile.getResult());
        Long campaignId = Long.valueOf(importInfo.getCampaignId());
        ExcelFileKey fileKey = saveToMds(clientId, campaignId, importInfo.getCampaignName(), excelFile);

        return Result.successful(importInfo.withExcelFileKey(fileKey));
    }

    /**
     * Возвращает необходимые параметры групп и объявлений
     * Необходимость выгрузки параметров групп определяется по флагу {@param fetchAdGroupInfo}
     * а необходимость для объявлений определяется по полученным объявлениям в {@param dataFromExcelFile#getAdsSheets()}
     */
    private InternalAdImportInfo getImportInfo(ClientId clientId, Set<ObjectType> objectTypesForRead,
                                               ExcelFetchedData dataFromExcelFile) {
        Long campaignId = dataFromExcelFile.getAdGroupsSheet().getSheetDescriptor().getCampaignId();
        Campaign campaign = campaignService.getCampaigns(clientId, Set.of(campaignId)).get(0);
        Map<Long, Long> placeIdByCampaignId = campaignService.getCampaignInternalPlaces(clientId, Set.of(campaignId));
        //noinspection ConstantConditions
        String userLogin = userService.getUser(campaign.getUserId()).getLogin();

        List<AdGroupInfo> adGroupsInfo = objectTypesForRead.contains(ObjectType.AD_GROUP)
                ? getAdGroupsInfo(dataFromExcelFile.getAdGroupsSheet()) : null;
        List<AdInfo> adsInfo = objectTypesForRead.contains(ObjectType.AD)
                ? getAdsInfo(clientId, dataFromExcelFile.getAdsSheets()) : null;

        return new InternalAdImportInfo()
                .withClientLogin(userLogin)
                .withCampaignId(campaign.getId().toString())
                .withCampaignName(campaign.getName())
                .withCampaignArchived(campaign.getStatusArchived())
                .withCampaignType(campaign.getType())
                .withPlaceId(placeIdByCampaignId.get(campaignId).toString())
                .withAdGroups(adGroupsInfo)
                .withAds(adsInfo)
                .withAdsCount(ifNotNull(adsInfo, List::size));
    }

    private List<AdInfo> getAdsInfo(ClientId clientId,
                                    List<ExcelSheetFetchedData<InternalBannerRepresentation>> adsSheets) {
        if (CollectionUtils.isEmpty(adsSheets)) {
            return Collections.emptyList();
        }

        var templateResources = getAdsTemplateResources(adsSheets);
        List<TemplateVariable> imageTemplateVariables = getImageTemplateVariables(adsSheets, templateResources);
        Set<Long> imageTemplateResourceIds = listToSet(imageTemplateVariables, TemplateVariable::getTemplateResourceId);

        Set<String> imageHashes = getTemplateVariableImageHashes(imageTemplateVariables);
        List<BannerImageFormat> bannerImageFormats = imageService.getBannerImageFormats(clientId, imageHashes);
        Map<String, BannerImageFormat> bannerImageFormatsByHash =
                listToMap(bannerImageFormats, BannerImageFormat::getImageHash);

        Map<String, BannerImageFromPool> bannerImageFromPoolsByHashes =
                imageService.getBannerImageFromPoolsByHashes(clientId, imageHashes);

        return toAdsInfo(adsSheets, templateResources,
                bannerImageFormatsByHash, bannerImageFromPoolsByHashes, imageTemplateResourceIds);
    }

    private List<ReadOnlyDirectTemplateResource> getAdsTemplateResources(
            List<ExcelSheetFetchedData<InternalBannerRepresentation>> adsSheets) {
        Set<Long> templateIds = listToSet(adsSheets, sheet -> sheet.getSheetDescriptor().getTemplateId());
        return StreamEx.of(templateResourceService.getReadonlyByTemplateIds(templateIds)
                .values())
                .sorted(Comparator.comparing(ReadOnlyDirectTemplateResource::getId))
                .toList();
    }

    private ExcelFileKey saveToMds(ClientId clientId, Long campaignId, String campaignName, MultipartFile excelFile) {
        try (TraceProfile profile = Trace.current().profile("internal_ad_export:save_to_mds")) {
            String mdsPath = generateMdsPath(clientId, campaignId, campaignName);
            MdsPostResponse response = mdsHolder
                    .upload(mdsPath, new ByteArrayInputStreamSource(excelFile.getBytes()), EXPIRE_DURATION);

            return ExcelFileKey.create(response.getKey().getFilename(), response.getKey().getGroup());
        } catch (IOException e) {
            logger.error("got IOException when getBytes of excelFile", e);
            throw new RuntimeException(e);
        }
    }

    String saveToMds(ClientId clientId, CampaignInfoWithWorkbook campaignInfoWithWorkbook) {
        try (TraceProfile profile = Trace.current().profile("internal_ad_export:save_to_mds")) {
            byte[] data = toBytes(campaignInfoWithWorkbook.getWorkbook());

            String mdsPath = generateMdsPath(clientId, campaignInfoWithWorkbook.getCampaignId(),
                    campaignInfoWithWorkbook.getCampaignName());
            MdsPostResponse response = mdsHolder.upload(mdsPath, new ByteArrayInputStreamSource(data), EXPIRE_DURATION);

            return MdsUrlUtils.actionUrl(mdsHolder.getHosts().getHostPortForRead(),
                    "get", mdsHolder.getNamespace().getName(), response.getKey());
        }
    }

    /**
     * Импорт данных из excel файла
     * Возвращает результат импорта и урл на полученный excel файл с хэшами новых загруженных изображений
     * Урл возвращается если результат импорта был успешен
     */
    public Result<ExcelImportResult> importInternal(Long operatorUid, UidAndClientId uidAndClientId,
                                                    InternalAdImportRequest request) {
        ValidationResult<InternalAdImportRequest, Defect> vr =
                InternalAdValidationService.validateImportRequest(request);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }

        MdsFileKey fileKey = toMdsFileKey(request.getExcelFileKey());
        InputStream input = downloadExcelFileFromMds(mdsHolder, fileKey);

        Set<ObjectType> objectTypesForImport = importModeToObjectTypes(request.getImportMode());
        Result<ExcelFetchedData> dataFromExcelFile = internalAdExcelService
                .getAndValidateDataFromExcelFile(uidAndClientId.getClientId(), input, objectTypesForImport);
        if (ValidationUtils.hasValidationIssues(dataFromExcelFile)) {
            return Result.broken(dataFromExcelFile.getValidationResult());
        }

        var adsSheets = dataFromExcelFile.getResult().getAdsSheets();
        boolean hasNewUploadedImages = isNotEmpty(adsSheets) && isNotEmpty(request.getAdsImages())
                && objectTypesForImport.contains(ObjectType.AD);
        if (hasNewUploadedImages) {
            enrichUploadedImageHashes(uidAndClientId.getClientId(), adsSheets, request.getAdsImages());
        }

        // берем новые группы до вызова операции, после вызова для них будут проставлены id
        List<InternalAdGroup> adGroupsForAdd = getAdGroupsForAdd(dataFromExcelFile.getResult().getAdGroupsSheet());

        ExcelImportResult result = internalAdExcelImportService.importFromExcel(operatorUid, uidAndClientId,
                request.getOnlyValidation(), objectTypesForImport, dataFromExcelFile.getResult());

        // если есть новые картинки и мы находимся на третьем шаге
        boolean needUploadUpdatedFile = hasNewUploadedImages && request.getOnlyValidation();
        String url = uploadIfNeedExcelFileAndGetUrl(needUploadUpdatedFile,
                mdsHolder, fileKey, uidAndClientId.getClientId(), dataFromExcelFile.getResult(), false);
        result.withExcelFileUrl(url);

        return Result.successful(result);
    }

    /**
     * Возвращает урл на MDS, где лежит excel-файл с обновленными данными:
     * хэшами новых загруженных изображений и/или id созданных групп
     * или если обновлений по данным нет, то возвращается урл на файл, который получали для импорта
     */
    String uploadIfNeedExcelFileAndGetUrl(boolean needUpload, MdsHolder mdsHolder, MdsFileKey fileKey,
                                          ClientId clientId, ExcelFetchedData excelFetchedData,
                                          boolean hideEmptyColumns) {
        if (needUpload) {
            CampaignInfoWithWorkbook campaignInfoWithWorkbook;
            try (TraceProfile profile = Trace.current().profile("internal_ad_import:create_workbook")) {
                campaignInfoWithWorkbook = internalAdExcelService.createWorkbook(clientId, excelFetchedData,
                        hideEmptyColumns);
            }

            return saveToMds(clientId, campaignInfoWithWorkbook);
        }

        return MdsUrlUtils.actionUrl(mdsHolder.getHosts().getHostPortForRead(),
                "get", mdsHolder.getNamespace().getName(), fileKey);
    }

    /**
     * Возвращает все картиночные переменные всех объявлений
     */
    private List<TemplateVariable> getImageTemplateVariables(
            List<ExcelSheetFetchedData<InternalBannerRepresentation>> adsSheets,
            List<ReadOnlyDirectTemplateResource> templateResources) {
        Set<Long> imageTemplateResourceIds = filterAndMapToSet(templateResources,
                ReadOnlyDirectTemplateResource::isBananaImage, ReadOnlyDirectTemplateResource::getId);

        List<InternalBannerRepresentation> adRepresentations = flatMap(adsSheets, ExcelSheetFetchedData::getObjects);
        return StreamEx.of(adRepresentations)
                .map(representation -> representation.getBanner().getTemplateVariables())
                .flatMap(Collection::stream)
                .filter(templateVariable -> imageTemplateResourceIds.contains(templateVariable.getTemplateResourceId()))
                .toImmutableList();
    }

    /**
     * Добавляет для картиночных переменных объявлений хэши новых загруженных изображений
     * Переменная для которой нужно добавить хэш находится по связке fileName -> imageHash из {@param adsImages}
     * Если значение картиночной переменной не найдется в базе и в {@param adsImages},
     * то будет заменено на {@link InternalAdExcelWebService#UNKNOWN_IMAGE_HASH}
     */
    void enrichUploadedImageHashes(ClientId clientId,
                                   List<ExcelSheetFetchedData<InternalBannerRepresentation>> adsSheets,
                                   List<UploadedImageInfo> adsImages) {
        var templateResources = getAdsTemplateResources(adsSheets);
        List<TemplateVariable> imageTemplateVariables = getImageTemplateVariables(adsSheets, templateResources);

        Set<String> imageHashes = getTemplateVariableImageHashes(imageTemplateVariables);
        List<BannerImageFormat> bannerImageFormats = imageService.getBannerImageFormats(clientId, imageHashes);
        Set<String> existingImageHashes = listToSet(bannerImageFormats, BannerImageFormat::getImageHash);

        Map<String, String> uploadedImageHashByFileName =
                listToMap(adsImages, UploadedImageInfo::getFileName, UploadedImageInfo::getImageHash);

        StreamEx.of(imageTemplateVariables)
                .filter(templateVariable -> templateVariable.getInternalValue() != null)
                .remove(templateVariable -> existingImageHashes.contains(templateVariable.getInternalValue()))
                .forEach(templateVariable -> templateVariable.setInternalValue(
                        uploadedImageHashByFileName
                                .getOrDefault(templateVariable.getInternalValue(), UNKNOWN_IMAGE_HASH)));
    }

}
