package ru.yandex.direct.web.entity.feed.controller

import com.fasterxml.jackson.annotation.JsonProperty
import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiParam
import io.swagger.annotations.ApiResponse
import io.swagger.annotations.ApiResponses
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.MediaType
import org.springframework.stereotype.Controller
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.ResponseBody
import org.springframework.web.multipart.MultipartFile
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.web.core.model.WebErrorResponse
import ru.yandex.direct.web.core.model.WebResponse
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN
import ru.yandex.direct.web.entity.feed.service.FeedWebService
import ru.yandex.direct.web.entity.feed.service.FeedWebService.AddFeedData
import ru.yandex.direct.web.entity.feed.service.FeedWebService.UpdateFeedData
import ru.yandex.direct.web.entity.freelancer.model.FreelancerAvatarUploadResponse

@Controller
@RequestMapping(value = ["/feed"], produces = [MediaType.APPLICATION_JSON_UTF8_VALUE])
@Api(tags = ["feed"])
open class FeedController @Autowired constructor(
    private val authenticationSource: DirectWebAuthenticationSource,
    private val feedWebService: FeedWebService
) {

    @ApiOperation(value = "addFeed", httpMethod = "POST", nickname = "addFeed")
    @ApiResponses(
        ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse::class),
        ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse::class),
        ApiResponse(code = 200, message = "Ok", response = FreelancerAvatarUploadResponse::class)
    )
    @PreAuthorizeWrite
    @PostMapping(path = ["/add_feed"], consumes = [MediaType.MULTIPART_FORM_DATA_VALUE])
    @ResponseBody
    open fun addFeed(@RequestParam(value = "name", required = true) name: String,
                     @ApiParam(name = "source", required = true, value = "Expected values: FILE, SITE, URL.")
                     @RequestParam(value = "source", required = true) source: String,
                     @ApiParam(name = "business_type", required = true, value = "Expected values: RETAIL, HOTELS, REALTY, AUTO, FLIGHTS, NEWS, OTHER.")
                     @RequestParam(value = "business_type", required = true) businessType: String,
                     @RequestParam(value = "url", required = false) url: String?,
                     @RequestParam(value = "login", required = false) login: String?,
                     @RequestParam(value = "password", required = false) password: String?,
                     @RequestParam(value = "is_remove_utm", required = false) isRemoveUtm: Boolean?,
                     @RequestParam(value = "file", required = false) file: MultipartFile?,
                     @RequestParam(value = PARAMETER_ULOGIN, required = false) `_`: String?): WebResponse {
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser
        val feedData = AddFeedData(name, source, businessType, url, login, password, isRemoveUtm, file)
        return feedWebService.addFeed(operator, subjectUser, feedData)
    }

    @ApiOperation(value = "updateFeed", httpMethod = "POST", nickname = "updateFeed")
    @ApiResponses(
        ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse::class),
        ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse::class),
        ApiResponse(code = 200, message = "Ok", response = FreelancerAvatarUploadResponse::class)
    )
    @PreAuthorizeWrite
    @PostMapping(path = ["/update_feed"], consumes = [MediaType.MULTIPART_FORM_DATA_VALUE])
    @ResponseBody
    open fun updateFeed(@RequestParam(value = "feed_id", required = true) feedId: Long,
                        @RequestParam(value = "name", required = false) name: String,
                        @RequestParam(value = "url", required = false) url: String?,
                        @RequestParam(value = "login", required = false) login: String?,
                        @RequestParam(value = "password", required = false) password: String?,
                        @RequestParam(value = "is_remove_utm", required = false) isRemoveUtm: Boolean?,
                        @RequestParam(value = "file", required = false) file: MultipartFile?,
                        @RequestParam(value = PARAMETER_ULOGIN, required = false) `_`: String?): WebResponse {
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser
        val feedData = UpdateFeedData(feedId, name, url, login, password, isRemoveUtm, file)
        return feedWebService.updateFeed(operator, subjectUser, feedData)
    }

    data class DeleteFeedRequest(@JsonProperty(value = "feed_ids", required = true) val feedIds: List<Long>)

    @ApiOperation(value = "deleteFeed", httpMethod = "POST", nickname = "deleteFeed")
    @ApiResponses(
        ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse::class),
        ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse::class),
        ApiResponse(code = 200, message = "Ok", response = FreelancerAvatarUploadResponse::class)
    )
    @PreAuthorizeWrite
    @PostMapping(path = ["/delete_feed"], consumes = [MediaType.APPLICATION_JSON_VALUE])
    @ResponseBody
    open fun deleteFeed(@RequestBody request: DeleteFeedRequest,
                        @RequestParam(value = PARAMETER_ULOGIN, required = false) `_`: String?): WebResponse {
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser
        return feedWebService.deleteFeed(operator, subjectUser, request.feedIds)
    }

    @ApiOperation(value = "refreshFeed", httpMethod = "POST", nickname = "refreshFeed")
    @ApiResponses(
        ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse::class),
        ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse::class),
        ApiResponse(code = 200, message = "Ok", response = FreelancerAvatarUploadResponse::class)
    )
    @PreAuthorizeWrite
    @PostMapping(path = ["/refresh_feed"], consumes = [MediaType.MULTIPART_FORM_DATA_VALUE])
    @ResponseBody
    open fun refreshFeed(@RequestParam(value = "feed_id", required = true) feedId: Long,
                         @RequestParam(value = PARAMETER_ULOGIN, required = false) `_`: String?): WebResponse {
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser

        return feedWebService.refreshFeed(operator, subjectUser, feedId)
    }
}
