package ru.yandex.direct.web.entity.feed.service

import ru.yandex.direct.core.entity.feed.model.BusinessType
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.model.Source
import ru.yandex.direct.core.entity.feed.model.Source.FILE
import ru.yandex.direct.core.entity.feed.model.Source.SITE
import ru.yandex.direct.core.entity.feed.model.Source.URL
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.web.entity.feed.service.FeedWebService.AddFeedData
import ru.yandex.direct.web.entity.feed.service.FeedWebService.DataConvertException
import ru.yandex.direct.web.entity.feed.service.FeedWebService.UpdateFeedData

const val MAX_OFFERS_SIZE_PER_CLIENT = 2000L

fun convertToFeed(feedData: AddFeedData): Feed {
    val feed = Feed().apply {
        name = feedData.name
        businessType = getEnumValue<BusinessType>(feedData.businessType)
                ?: throw DataConvertException("Unknown businessType '${feedData.businessType}'")
    }
    when (getEnumValue<Source>(feedData.source)) {
        FILE -> feed.apply {
            source = FILE
            fileData = feedData.file?.bytes
                    ?: throw DataConvertException("File is required for source 'FILE'")
            filename = feedData.file.originalFilename
        }
        SITE -> feed.apply {
            source = SITE
            url = feedData.url
        }
        URL -> feed.apply {
            source = URL
            url = feedData.url
            isRemoveUtm = feedData.removeUtm
            login = feedData.login
            plainPassword = feedData.password
        }
        else -> {
            throw DataConvertException("Unknown source '${feedData.source}'")
        }
    }
    return feed
}

fun convertToFeedChanges(feedData: UpdateFeedData): ModelChanges<Feed> {
    val changes = ModelChanges(feedData.feedId, Feed::class.java)
    changes.processNotNull(feedData.name, Feed.NAME)
    changes.processNotNull(feedData.url, Feed.URL)
    changes.processNotNull(feedData.removeUtm, Feed.IS_REMOVE_UTM)
    if (feedData.login != null) {
        changes.process(feedData.login.ifEmpty { null }, Feed.LOGIN)
    }
    if (feedData.password != null) {
        changes.process(feedData.password.ifEmpty { null }, Feed.PLAIN_PASSWORD)
    }
    changes.processNotNull(feedData.file?.originalFilename, Feed.FILENAME)
    changes.processNotNull(feedData.file?.bytes, Feed.FILE_DATA)
    return changes
}

inline fun <reified T : Enum<T>> getEnumValue(valueName: String): T? =
        T::class.java.enumConstants.firstOrNull { it.name == valueName }
