package ru.yandex.direct.web.entity.feed.service

import com.fasterxml.jackson.annotation.JsonProperty
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import org.springframework.web.multipart.MultipartFile
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.entity.feed.validation.FeedDefectIds
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.core.validation.defects.params.ModelIdDefectParams
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.result.Result
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.PathHelper.field
import ru.yandex.direct.validation.result.PathHelper.index
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.web.core.model.WebErrorResponse
import ru.yandex.direct.web.core.model.WebResponse
import ru.yandex.direct.web.entity.SuccessResponse
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService
import ru.yandex.direct.web.validation.model.WebValidationResult

@Service
class FeedWebService @Autowired constructor(
    private val feedService: FeedService,
    private val validationResultConversionService: ValidationResultConversionService
) {
    data class AddFeedData(val name: String,
                           val source: String,
                           val businessType: String,
                           val url: String?,
                           val login: String?,
                           val password: String?,
                           val removeUtm: Boolean?,
                           val file: MultipartFile?)

    data class UpdateFeedData(val feedId: Long,
                              val name: String?,
                              val url: String?,
                              val login: String?,
                              val password: String?,
                              val removeUtm: Boolean?,
                              val file: MultipartFile?)

    data class DataConvertException(override val message: String) : RuntimeException(message)

    data class FeedResult(@JsonProperty("feed_id") val feedId: Long)
    data class FeedChangeResponse(private val feedId: Long) : SuccessResponse<FeedResult>() {
        init {
            super.setResult(FeedResult(feedId))
        }
    }

    data class FeedMassChangeResponse(
        private val isSuccessful: Boolean,
        @JsonProperty("validation_result") val webValidationResult: WebValidationResult,
        @JsonProperty("result") val result: List<Long>,
    ) : WebResponse {
        override fun isSuccessful() = isSuccessful
    }

    fun addFeed(operator: User, subjectUser: User, feedData: AddFeedData): WebResponse {
        val feed: Feed = try {
            convertToFeed(feedData)
        } catch (e: DataConvertException) {
            return WebErrorResponse(e.message, null)
        }
        val massResult = feedService.addFeeds(subjectUser.clientId, subjectUser.uid, operator.uid, listOf(feed))
        return convertToResponse(massResult, true)
    }

    fun updateFeed(operator: User, subjectUser: User, feedData: UpdateFeedData): WebResponse {
        val changes: ModelChanges<Feed> = try {
            convertToFeedChanges(feedData)
        } catch (e: DataConvertException) {
            return WebErrorResponse(e.message, null)
        }
        val massResult = feedService.updateFeeds(subjectUser.clientId, subjectUser.uid, operator.uid, listOf(changes))
        return convertToResponse(massResult, false)
    }

    fun deleteFeed(operator: User, subjectUser: User, feedIds: List<Long>): WebResponse {
        val massResult = feedService.deleteFeeds(subjectUser.clientId, subjectUser.uid, operator.uid, feedIds)
        return convertToMassResponse(massResult)
    }

    fun refreshFeed(operator: User, subjectUser: User, feedId: Long): WebResponse {
        val result = feedService.refreshFeed(subjectUser.clientId, subjectUser.uid, operator.uid, feedId)
        return convertToResponse(result)
    }

    private fun convertToResponse(result: MassResult<Long>, replaceResultWithDuplicate: Boolean): WebResponse =
        result.let { massResult ->
            massResult.result?.let { it[0] }
                ?.takeIf { it.isSuccessful }
                ?.let { FeedChangeResponse(it.result) }
                ?: let {
                    val existentFeedId = if (replaceResultWithDuplicate) {
                        getExistentFeedIdFromValidationResult(massResult.validationResult)
                    } else null

                    if (existentFeedId != null) {
                        FeedChangeResponse(existentFeedId)
                    } else {
                        validationResultConversionService.buildValidationResponse(massResult.validationResult)
                    }
                }
        }

    private fun convertToResponse(result: Result<Long>): WebResponse =
        result.let { r ->
            r.takeIf { it.isSuccessful }?.let { FeedChangeResponse(it.result) }
                ?: let {
                    validationResultConversionService.buildValidationResponse(r.validationResult)
                }
        }

    private fun convertToMassResponse(massResult: MassResult<Long>): WebResponse {
        val isSuccessful = massResult.toResultList().any { it.isSuccessful }
        val webValidationResult = validationResultConversionService.buildWebValidationResult(massResult.validationResult)
        val result = massResult.toResultList().map { it.result }.filterNotNull()
        return FeedMassChangeResponse(isSuccessful, webValidationResult, result)
    }

    private fun getExistentFeedIdFromValidationResult(vr: ValidationResult<*, Defect<*>>): Long? {
        val errors = vr.subResults[index(0)]
            ?.subResults?.get(field(Feed.URL.name()))
            ?.errors

        if (errors == null || errors.size != 1) {
            return null
        }

        val defect = errors[0]

        if (defect.defectId() != FeedDefectIds.ModelId.FEED_BY_SITE_WITH_SAME_DOMAIN_ALREADY_EXISTS) {
            return null
        }

        return (defect.params() as? ModelIdDefectParams)?.id
    }

}
