package ru.yandex.direct.web.entity.feedback.service;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.ExecutionException;
import java.util.function.Supplier;

import com.amazonaws.SdkClientException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.google.protobuf.Timestamp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.crm.mail.model.MailProtobuf.File;
import ru.yandex.crm.mail.model.MailProtobuf.Mail;
import ru.yandex.crm.mail.model.MailProtobuf.MailAddress;
import ru.yandex.crm.mail.model.MailProtobuf.MailRequest;
import ru.yandex.crm.mail.model.MailProtobuf.User;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.kikimr.persqueue.LogbrokerClientFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.kikimr.persqueue.producer.async.AsyncProducerConfig;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;

@Service
public class CommanderFeedbackService {
    private static final Logger logger = LoggerFactory.getLogger(CommanderFeedbackService.class);

    private static final String DEFAULT_MAIL_FROM = "support@direct.yandex.ru";
    private static final String DEFAULT_MAIL_TO = "commander@support.yandex.ru";
    private static final String DEFAULT_SUBJECT = "Direct Commander Feedback";

    static final String SCREENSHOT_BASE_NAME = "screenshot.png";
    static final String SCREENSHOT_TYPE = "image/png";
    static final String LOGDATA_BASE_NAME = "logs.txt";
    static final String LOGDATA_TYPE = "text/plain";

    private final TvmIntegration tvmIntegration;
    private final CrmTransportConfig config;
    private final AmazonS3 amazonS3;

    private final LogbrokerClientFactory logbrokerClientFactory;

    @Autowired
    public CommanderFeedbackService(TvmIntegration tvmIntegration, CrmTransportConfig config, AmazonS3 amazonS3) {
        this.tvmIntegration = tvmIntegration;
        this.config = config;
        this.amazonS3 = amazonS3;
        logbrokerClientFactory = new LogbrokerClientFactory(new ProxyBalancer(config.getLogbrokerHost()));
    }

    CommanderFeedbackService(TvmIntegration tvmIntegration, CrmTransportConfig config, AmazonS3 amazonS3,
                             LogbrokerClientFactory logbrokerClientFactory) {
        this.tvmIntegration = tvmIntegration;
        this.config = config;
        this.amazonS3 = amazonS3;
        this.logbrokerClientFactory = logbrokerClientFactory;
    }

    public WebResponse sendMessage(String login, String email, String text, byte[] screenshot, byte[] logdata) {
        Instant time = Instant.now().truncatedTo(ChronoUnit.SECONDS);

        AsyncProducerConfig producerConfig = createProducerConfig(config, tvmIntegration);

        String screenshotPath = uploadFileToMdsS3(
                makeFullName(SCREENSHOT_BASE_NAME, login, time), SCREENSHOT_TYPE, screenshot);
        String logPath = uploadFileToMdsS3(
                makeFullName(LOGDATA_BASE_NAME, login, time), LOGDATA_TYPE, logdata);

        try (AsyncProducer producer = logbrokerClientFactory.asyncProducer(producerConfig)) {

            var initResponse = producer.init().get();
            logger.info("Initiated write session {} to logbroker topic {}",
                    initResponse.getSessionId(), initResponse.getTopic());

            MailRequest request = makeRequest(login, email, text, screenshotPath, logPath, time);
            logger.info("Sending request {}", request);

            var writeResponse = producer.write(request.toByteArray()).get();
            logger.info("Successfully written request");

            return new WebSuccessResponse();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException("Failed to write request", e);
        } catch (ExecutionException e) {
            throw new RuntimeException("Failed to write request", e);
        }
    }

    private static AsyncProducerConfig createProducerConfig(CrmTransportConfig config, TvmIntegration tvmIntegration) {
        Supplier<Credentials> credentialsProvider = createCredentialsProvider(config, tvmIntegration);
        return AsyncProducerConfig.builder(config.getLogbrokerTopic(), config.getLogbrokerSourceId().getBytes())
                .setCredentialsProvider(credentialsProvider).build();
    }

    private static Supplier<Credentials> createCredentialsProvider(
            CrmTransportConfig config,
            TvmIntegration tvmIntegration
    ) {
        TvmService logbrokerTvmService = TvmService.fromStringStrict(config.getLogbrokerTvmServiceName());
        return () -> {
            String serviceTicket = tvmIntegration.getTicket(logbrokerTvmService);
            return Credentials.tvm(serviceTicket);
        };
    }

    private static MailRequest makeRequest(String login, String email, String text,
                                           String screenshotPath, String logPath, Instant time) {
        return MailRequest.newBuilder()
                .setUser(makeUser(login))
                .setMail(makeMail(email, text, screenshotPath, logPath, time))
                .build();
    }

    private static Mail makeMail(String email, String text, String screenshotPath, String logPath, Instant time) {
        var builder = Mail.newBuilder();
        builder.setFrom(makeMailAddress(email != null ? email : DEFAULT_MAIL_FROM));
        builder.addTo(makeMailAddress(DEFAULT_MAIL_TO));
        builder.setSubject(DEFAULT_SUBJECT);
        builder.setText(text);
        builder.setTime(makeTimeStamp(time));
        if (screenshotPath != null) {
            builder.addFiles(makeFile(SCREENSHOT_BASE_NAME, screenshotPath));
        }
        if (logPath != null) {
            builder.addFiles(makeFile(LOGDATA_BASE_NAME, logPath));
        }
        return builder.build();
    }

    private static MailAddress makeMailAddress(String email) {
        return MailAddress.newBuilder().setEmail(email).build();
    }

    private static User makeUser(String login) {
        return User.newBuilder().setLogin(login).build();
    }

    private static Timestamp makeTimeStamp(Instant time) {
        return Timestamp.newBuilder()
                .setSeconds(time.getEpochSecond())
                .build();
    }

    private static File makeFile(String name, String path) {
        return File.newBuilder()
                .setName(name)
                .setUrlPath(path)
                .build();
    }

    private static String makeFullName(String base, String login, Instant time) {
        return base + "_" + login + "_" + time;
    }

    private String uploadFileToMdsS3(String name, String contentType, byte[] data) {
        if (data == null) {
            return null;
        }
        try (ByteArrayInputStream is = new ByteArrayInputStream(data)) {
            ObjectMetadata metadata = new ObjectMetadata();
            metadata.setContentLength(data.length);
            metadata.setContentType(contentType);
            amazonS3.putObject(config.getMdsBucket(), name, is, metadata);
            var path = amazonS3.getUrl(config.getMdsBucket(), name).toString();
            logger.info("Uploaded {} to MDS-S3 by {}", name, path);
            return path;
        } catch (IOException | SdkClientException e) {
            logger.error("Failed to upload {} to MDS-S3", name);
            return null;
        }
    }
}
