package ru.yandex.direct.web.entity.feedback.service;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.mail.AuthorizedMailMessage;
import ru.yandex.direct.mail.EmailAddress;
import ru.yandex.direct.mail.MailSender;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.feedback.model.FeedbackRequest;

/**
 * Сервис для обработки обратной связи из интерфейса. Отправляет письмо с сообщением от пользователя
 * и дополнительными данными на внутренний адрес для создания CRM тикета
 */
@Service
@ParametersAreNonnullByDefault
public class FeedbackService {
    private static final Logger logger = LoggerFactory.getLogger(FeedbackService.class);
    private static final String GRID_FEEDBACK_EMAIL_SUBJECT = "Обратная связь об интерфейсе Директа";
    private static final String GRID_FEEDBACK_OTRS_HEADER_TEXT = "X-OTRS-fromfeedback direct-new-interface";
    private static final String GRID_FEEDBACK_EMAIL_TO = "client-service@yandex-team.ru";
    private static final String CALLTRACKING_FEEDBACK_PREFIX = "Обратная связь про Calltracking: ";

    private final DirectWebAuthenticationSource authenticationSource;
    private final MailSender mailSender;

    @Autowired
    public FeedbackService(DirectWebAuthenticationSource authenticationSource, MailSender mailSender) {
        this.authenticationSource = authenticationSource;
        this.mailSender = mailSender;
    }

    public WebResponse sendGridFeedback(FeedbackRequest request) {
        User operator = authenticationSource.getAuthentication().getOperator();
        logger.info("Got feedback from {}", operator.getLogin());
        // Не отправляем фидбэк на Calltracking в поддержку (DIRECT-131608)
        boolean skipSendingEmail = Boolean.TRUE.equals(request.getIsCalltrackingFeedback())
                || Boolean.FALSE.equals(request.getSendToCrm()); // сказали не отправлять

        // также не отправляем фидбек, если прислали только оценку, достаточно записи в логе cmd
        if (skipSendingEmail || request.getMessage() == null) {
            return new WebSuccessResponse();
        }

        User subjectUser = authenticationSource.getAuthentication().getSubjectUser();

        String message = buildMessageFromRequest(request);
        String gridEmailBody = getGridEmailBody(operator, message, request);
        EmailAddress addressTo = new EmailAddress(GRID_FEEDBACK_EMAIL_TO, "CRM");
        sendMailFromUser(operator, subjectUser, addressTo, GRID_FEEDBACK_EMAIL_SUBJECT, gridEmailBody);

        return new WebSuccessResponse();
    }

    private String buildMessageFromRequest(FeedbackRequest request) {
        return Boolean.TRUE.equals(request.getIsCalltrackingFeedback())
                ? CALLTRACKING_FEEDBACK_PREFIX + request.getMessage()
                : request.getMessage();
    }

    private void sendMailFromUser(User operator, User subjectUser,
                                  EmailAddress addressTo, String subject, String messageBody) {
        var addressFrom = new EmailAddress(operator.getEmail(), operator.getLogin());
        Long operatorUid = operator.getUid();
        Long subjectUserClientId = subjectUser.getClientId().asLong();

        var mailMessage = new AuthorizedMailMessage(operatorUid, subjectUserClientId,
                addressFrom, addressTo, subject, messageBody);
        mailSender.send(mailMessage);
    }

    private String getGridEmailBody(User user, String message, FeedbackRequest request) {
        var builder = new StringBuilder();
        var url = request.getUrl();
        var role = request.getRole();
        var experience = request.getExperience();
        var crmTag = request.getCrmTag();
        builder.append("Сообщение: ")
                .append(message)
                .append("\nСтраница, с которой отправлено сообщение: ")
                .append(url)
                .append("\nВремя отправки сообщения: ")
                .append(LocalDateTime.now().format(DateTimeFormatter.ISO_DATE_TIME))
                .append("\nЛогин: ")
                .append(user.getLogin())
                .append("\nClientId: ")
                .append(user.getClientId())
                .append("\nUID: ")
                .append(user.getUid());
        if (role != null) {
            builder.append("\nРоль в Директе: ")
                    .append(role);
        }
        if (experience != null) {
            builder.append("\nОпыт работы с Директом: ")
                    .append(experience);
        }
        builder.append("\n")
                .append(GRID_FEEDBACK_OTRS_HEADER_TEXT);
        if (crmTag != null) {
            builder.append('#')
                    .append(crmTag);
        }
        return builder.toString();
    }
}
