package ru.yandex.direct.web.entity.frontpage.controller;

import io.swagger.annotations.Api;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.MappingPathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.frontpage.model.CpmYndxFrontpageGeoWarningsResponse;
import ru.yandex.direct.web.entity.frontpage.model.CpmYndxFrontpagePriceWarningsResponse;
import ru.yandex.direct.web.entity.frontpage.model.CpmYndxFrontpagePriceWarningsResponseBuilder;
import ru.yandex.direct.web.entity.frontpage.model.CpmYndxFrontpageWarningsRequest;
import ru.yandex.direct.web.entity.frontpage.model.FrontpageGeoWarningsGetItem;
import ru.yandex.direct.web.entity.frontpage.model.FrontpageWarningsGetItem;
import ru.yandex.direct.web.entity.frontpage.service.CpmYndxFrontpageWarningsService;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static ru.yandex.direct.validation.result.PathHelper.path;
import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;

/**
 * Контроллер на выдачу ворнингов со страницы мультиредактирования группы в случае группы на главной
 */
@RestController
@RequestMapping(path = "cpmyndxfrontpage",
        consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "cpmYndxFrontpage")
public class CpmYndxFrontpageWarningsController {

    private static final MappingPathNodeConverter FRONTPAGE_GEO_WARNINGS_GET_ITEM_CONVERTER =
            MappingPathNodeConverter.builder("CpmYndxFrontpagePathConverters").build();

    private final ValidationResultConversionService validationResultConversionService;
    private final PathNodeConverterProvider pathNodeConverterProvider;
    private final DirectWebAuthenticationSource authenticationSource;
    private final CpmYndxFrontpageWarningsService cpmYndxFrontpageWarningsService;

    public CpmYndxFrontpageWarningsController(
            ValidationResultConversionService validationResultConversionService,
            PathNodeConverterProvider pathNodeConverterProvider,
            DirectWebAuthenticationSource authenticationSource,
            CpmYndxFrontpageWarningsService cpmYndxFrontpageWarningsService) {
        this.validationResultConversionService = validationResultConversionService;
        this.pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(FrontpageWarningsGetItem.class, FRONTPAGE_GEO_WARNINGS_GET_ITEM_CONVERTER)
                .fallbackTo(pathNodeConverterProvider)
                .build();
        this.authenticationSource = authenticationSource;
        this.cpmYndxFrontpageWarningsService = cpmYndxFrontpageWarningsService;
    }

    /**
     * Принимает на вход идентификатор кампании и единицу FrontpageGeoWarningsGetItem, который состоит из цены и гео
     * По переданным параметрам возвращает ворнинги и ошибки, возникающие при валидации цены
     *
     * @param frontpageGeoWarningsGetItem единица FrontpageGeoWarningsGetItem (гео и цена)
     * @param campaignId                  идентификатор кампании
     * @param subjectLogin                логин клиента
     */
    @RequestMapping(path = "/cpm/getFrontpageGeoWarnings", method = RequestMethod.POST)
    public CpmYndxFrontpageGeoWarningsResponse getCpmYndxFrontpageGeoWarnings(
            @RequestBody FrontpageGeoWarningsGetItem frontpageGeoWarningsGetItem,
            @RequestParam(value = "campaign_id") Long campaignId,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        User subjectUser = authenticationSource.getAuthentication().getSubjectUser();
        ClientId clientId = subjectUser.getClientId();

        ValidationResult<FrontpageGeoWarningsGetItem, Defect> validationResult = cpmYndxFrontpageWarningsService
                .getFrontpageGeoWarnings(frontpageGeoWarningsGetItem, campaignId, clientId);

        return new CpmYndxFrontpageGeoWarningsResponse(validationResultConversionService.buildWebValidationResult(
                validationResult, path(), pathNodeConverterProvider));
    }


    /**
     * Принимает на вход запрос с гео групп, ставками, идентификатором кампании и дополнительной информацией
     * По переданным параметрам возвращает ворнинги и ошибки, возникающие при валидации цены
     *
     * @param frontpageWarningsRequest запрос
     * @param subjectLogin             логин клиента
     */
    @RequestMapping(path = "/getFrontpagePriceWarnings", method = RequestMethod.POST)
    public CpmYndxFrontpagePriceWarningsResponse getFrontpagePriceWarnings(
            @RequestBody CpmYndxFrontpageWarningsRequest frontpageWarningsRequest,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        User operator = authenticationSource.getAuthentication().getOperator();
        User subjectUser = authenticationSource.getAuthentication().getSubjectUser();
        ClientId clientId = subjectUser.getClientId();
        long operatorUid = operator.getUid();

        return cpmYndxFrontpageWarningsService.getFrontpageWarnings(frontpageWarningsRequest, clientId, operatorUid,
                new CpmYndxFrontpagePriceWarningsResponseBuilder(validationResultConversionService,
                        pathNodeConverterProvider));
    }
}
