package ru.yandex.direct.web.entity.frontpage.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.bids.validation.PriceValidator;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType;
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingSelection;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.frontpage.model.CpmYndxFrontpagePriceWarningsResponse;
import ru.yandex.direct.web.entity.frontpage.model.CpmYndxFrontpagePriceWarningsResponseBuilder;
import ru.yandex.direct.web.entity.frontpage.model.CpmYndxFrontpageWarningsRequest;
import ru.yandex.direct.web.entity.frontpage.model.FrontpageGeoWarningsGetItem;
import ru.yandex.direct.web.entity.frontpage.model.FrontpageWarningsGetItem;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.assertj.core.util.Preconditions.checkState;
import static ru.yandex.direct.core.entity.campaign.model.CpmYndxFrontpageShowTypeUtils.toFrontpageShowType;
import static ru.yandex.direct.multitype.entity.LimitOffset.maxLimited;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.entity.adgroup.converter.AdGroupConverterUtils.convertGeo;

/**
 * Сервис для получения ворнингов и ошибок, связанных с валидацией цены на главной
 */
@Service
@ParametersAreNonnullByDefault
public class CpmYndxFrontpageWarningsService {
    private static final Long FICTIONAL_CAMPAIGN_ID_VALUE = -1L;

    private final CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService;
    private final ShardHelper shardHelper;
    private final ClientService clientService;
    private final AdGroupService adGroupService;
    private final RetargetingService retargetingService;

    @Autowired
    public CpmYndxFrontpageWarningsService(CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService,
                                           ShardHelper shardHelper,
                                           ClientService clientService,
                                           AdGroupService adGroupService,
                                           RetargetingService retargetingService) {
        this.cpmYndxFrontpageCurrencyService = cpmYndxFrontpageCurrencyService;
        this.shardHelper = shardHelper;
        this.clientService = clientService;
        this.adGroupService = adGroupService;
        this.retargetingService = retargetingService;
    }

    /**
     * Формирует параметры для похода в CpmYndxFrontpageCurrencyService и подставляет полученную оттуда информацию по
     * группе
     * в PriceValidator. Возвращает полученный оттуда ValidationResult
     *
     * @param frontpageGeoWarningsGetItem гео группы + цена
     * @param campaignId                  идентификатор кампании
     * @param clientId                    идентификатор клиента
     */
    @Deprecated
    public ValidationResult<FrontpageGeoWarningsGetItem, Defect> getFrontpageGeoWarnings(
            FrontpageGeoWarningsGetItem frontpageGeoWarningsGetItem, Long campaignId, ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        AdGroupSimple adGroup = toAdGroup(frontpageGeoWarningsGetItem, campaignId);
        Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> adGroupRestrictions = cpmYndxFrontpageCurrencyService
                .getAdGroupIndexesToPriceDataMapByAdGroups(singletonList(adGroup), shard, clientId);
        ItemValidationBuilder<FrontpageGeoWarningsGetItem, Defect> validationResultBuilder =
                ItemValidationBuilder.<FrontpageGeoWarningsGetItem, Defect>of(frontpageGeoWarningsGetItem);
        validationResultBuilder.item(frontpageGeoWarningsGetItem.getPrice(), FrontpageGeoWarningsGetItem.Prop.PRICE)
                .checkBy(new PriceValidator(clientService.getWorkCurrency(clientId),
                        AdGroupType.CPM_YNDX_FRONTPAGE,
                        adGroupRestrictions.get(0)));
        return validationResultBuilder.getResult();
    }

    /**
     * Формирует параметры для похода в CpmYndxFrontpageCurrencyService и подставляет полученную оттуда информацию по
     * группе
     * в PriceValidator. Возвращает полученный оттуда массив ValidationResult по каждой из переданных групп
     * и минимальную допустимую цену для показа хоть одной группы
     *
     * @param frontpageWarningsRequest запрос, содержащий информацию о валидируемых группах и кампании
     * @param clientId                 идентификатор клиента
     * @param operatorUid              uid оператора
     * @param responseBuilder          способ построения ответа по минимальной цене и списку ValidationResult
     */
    public CpmYndxFrontpagePriceWarningsResponse getFrontpageWarnings(
            CpmYndxFrontpageWarningsRequest frontpageWarningsRequest,
            ClientId clientId, Long operatorUid,
            CpmYndxFrontpagePriceWarningsResponseBuilder responseBuilder) {
        //Формируем список frontpageGeoWarningsGetItems по данным из запроса
        fillMissingRequestFields(frontpageWarningsRequest);
        List<FrontpageWarningsGetItem> frontpageWarningsGetItems =
                combineFrontpageWarningsFromRequestAndDb(frontpageWarningsRequest, clientId, operatorUid);
        fillMissingWarningItemsFields(frontpageWarningsGetItems, frontpageWarningsRequest.getStrategyAutoPrice(),
                clientId, operatorUid);

        //Получаем информацию об ограничениях по цене по каждому item из frontpageGeoWarningsGetItems
        Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> adGroupsAndCampaignPriceRestrictions =
                getPriceRestrictions(frontpageWarningsGetItems, frontpageWarningsRequest.getCampaignId(),
                        frontpageWarningsRequest.getAllowedFrontpageType(), clientId);

        //Вычисляем минимально допустимые для показа групп цены
        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        Map<Integer, BigDecimal> adGroupMinPrices = EntryStream.of(adGroupsAndCampaignPriceRestrictions)
                .mapValues(t -> t.getCpmYndxFrontpageMinPrice())
                .mapValues(price -> price.max(clientCurrency.getMinCpmPrice()))
                .toMap();

        //Вычисляем минимальную для показа хотя бы одной группы цену. Если групп нет, берётся цена только по данным о
        // кампании
        BigDecimal commonMinPrice = adGroupMinPrices.values()
                .stream()
                .min(Comparator.naturalOrder())
                .orElse(clientCurrency.getMinCpmPrice());

        //Строим список результатов валидации каждой из переданных групп
        List<ValidationResult<FrontpageWarningsGetItem, Defect>> validationResults =
                EntryStream.of(frontpageWarningsGetItems)
                        .mapKeyValue((index, warningItem) -> validateWarningItem(
                                adGroupsAndCampaignPriceRestrictions.get(index),
                                warningItem, clientCurrency))
                        .toList();
        checkState(validationResults.size() == adGroupMinPrices.size(),
                "По каждой из групп с результатом валидации должна быть найдена минимальная цена");

        return responseBuilder
                .withCommonMinPrice(commonMinPrice)
                .withAdGroupMinPrices(adGroupMinPrices)
                .withValidationResults(validationResults)
                .build();
    }

    /**
     * Заполнение не переданных в запросе данных дефолтными или фиктивными значениями
     */
    private void fillMissingRequestFields(CpmYndxFrontpageWarningsRequest frontpageWarningsRequest) {
        if (frontpageWarningsRequest.getCampaignGeo() == null) {
            frontpageWarningsRequest.setCampaignGeo("");
        }
        //Описанная ниже ситуация возможнав в том и только том случае, когда создаётся новая кампания
        //Устанавливаем ей фиктивный id, можно брать любой Long
        //Для этого случая не должны ходить в базу за группами кампании (их же нет), и передано их тоже не должно быть
        if (frontpageWarningsRequest.getCampaignId() == null) {
            frontpageWarningsRequest.setCampaignId(FICTIONAL_CAMPAIGN_ID_VALUE);
            if (frontpageWarningsRequest.getAllowedFrontpageType() == null) {
                frontpageWarningsRequest.setAllowedFrontpageType("frontpage,frontpage_mobile,browser_new_tab");
            }
            checkState(!frontpageWarningsRequest.getUseDbAdGroups() &&
                            isEmpty(frontpageWarningsRequest.getFrontpageWarningsGetItems()),
                    "Не должны брать никаких групп при непереданном идентификаторе кампании");
        }
    }

    /**
     * Формирование списка FrontpageGeoWarningsGetItemNew с информацией о валидируемых группах по данным из запроса
     * Для этого берём явный список FrontpageGeoWarningsGetItemNew из запроса, добавляем к нему группы
     * из базы(при переданном параметре get_adgroups_data) и добавляет один item с данными по кампании, если групп нет
     */
    private List<FrontpageWarningsGetItem> combineFrontpageWarningsFromRequestAndDb(
            CpmYndxFrontpageWarningsRequest frontpageWarningsRequest, ClientId clientId, Long operatorUid) {
        List<FrontpageWarningsGetItem> warningItems =
                nvl(frontpageWarningsRequest.getFrontpageWarningsGetItems(), new ArrayList<>());
        if (frontpageWarningsRequest.getUseDbAdGroups()) {
            List<FrontpageWarningsGetItem> adGroupDataItems = fromDb(frontpageWarningsRequest.getCampaignId(),
                    clientId, operatorUid);
            warningItems.addAll(adGroupDataItems);
        }
        if (frontpageWarningsRequest.getValidateCampaign() && warningItems.isEmpty()) {
            FrontpageWarningsGetItem campaignWarningItem =
                    new FrontpageWarningsGetItem(frontpageWarningsRequest.getStrategyAutoPrice(),
                            frontpageWarningsRequest.getCampaignGeo(), null);
            warningItems.add(campaignWarningItem);
        }
        return warningItems;
    }

    /**
     * Получение из базы списка FrontpageGeoWarningsGetItemNew с информацией по группам данной кампании
     */
    private List<FrontpageWarningsGetItem> fromDb(Long campaignId, ClientId clientId,
                                                  Long operatorUid) {
        List<TargetInterest> targetInterests = retargetingService.getRetargetings(
                new RetargetingSelection().withCampaignIds(singletonList(campaignId)),
                clientId, operatorUid, maxLimited());
        return targetInterests.stream()
                .map(t -> new FrontpageWarningsGetItem(t.getPriceContext(), null, t.getAdGroupId()))
                .collect(Collectors.toList());
    }

    /**
     * Заполнение пропущенных параметров групп значениями из базы
     *
     * @param frontpageWarningsGetItems список frontpageWarningsGetItem, в которых и заполняем поля
     * @param strategyAutoPrice         цена автостратегии, при наличии таковой
     * @param clientId                  идентификатор клиента
     * @param operatorUid               uid оператора
     */
    private void fillMissingWarningItemsFields(List<FrontpageWarningsGetItem> frontpageWarningsGetItems,
                                               @Nullable BigDecimal strategyAutoPrice, ClientId clientId, Long
                                                       operatorUid) {
        fillAdGroupGeo(frontpageWarningsGetItems, clientId);
        fillAdGroupPrices(frontpageWarningsGetItems, strategyAutoPrice, clientId, operatorUid);
    }

    /**
     * Заполнение пропущенного гео групп
     */
    private void fillAdGroupGeo(List<FrontpageWarningsGetItem> frontpageWarningsGetItems, ClientId clientId) {
        Set<Long> adGroupIds = frontpageWarningsGetItems.stream()
                .map(FrontpageWarningsGetItem::getAdGroupId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        if (adGroupIds.isEmpty()) {
            return;
        }
        Map<Long, AdGroupSimple> adGroups = adGroupService.getSimpleAdGroups(clientId, adGroupIds);
        frontpageWarningsGetItems.forEach(t -> {
            if (t.getGeo() == null && t.getAdGroupId() != null) {
                t.setGeo(adGroups.get(t.getAdGroupId()).getGeo()
                        .stream()
                        .map(String::valueOf)
                        .collect(Collectors.joining(",")));
            }
        });
    }

    /**
     * Заполнение пропущенных ставок групп
     */
    private void fillAdGroupPrices(List<FrontpageWarningsGetItem> frontpageWarningsGetItems,
                                   @Nullable BigDecimal strategyAutoPrice, ClientId clientId, Long operatorUid) {
        Set<Long> adGroupIds = frontpageWarningsGetItems.stream()
                .map(FrontpageWarningsGetItem::getAdGroupId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        if (adGroupIds.isEmpty()) {
            return;
        }
        List<TargetInterest> targetInterests = retargetingService.getRetargetings(
                new RetargetingSelection().withAdGroupIds(new ArrayList<>(adGroupIds)),
                clientId, operatorUid, maxLimited());
        Map<Long, BigDecimal> priceByPid = targetInterests
                .stream()
                .filter(t -> t.getPriceContext() != null)
                .collect(Collectors.toMap(t -> t.getAdGroupId(), t -> t.getPriceContext(), (p1, p2) -> p1.max(p2)));
        frontpageWarningsGetItems.forEach(t -> {
            if (t.getPrice() == null && t.getAdGroupId() != null) {
                t.setPrice(priceByPid.get(t.getAdGroupId()));
            }
        });

        //Если автостратегия, ставки на группе не имеют смысла, используем ставку со стратегии
        if (strategyAutoPrice != null) {
            frontpageWarningsGetItems.forEach(t -> t.setPrice(strategyAutoPrice));
        }
    }

    /**
     * Ходим в СpmYndxFrontpageCurrencyService и по сформированному списку frontpageGeoWarningsGetItems
     * получаем список элементов CpmYndxFrontpageAdGroupPriceRestrictions с ограничениями на минимальную ставку групп,
     * соответствующих frontpageWarningsGetItems
     */
    private Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> getPriceRestrictions(
            List<FrontpageWarningsGetItem> frontpageWarningsGetItems, Long campaignId,
            String allowedFrontpageType, ClientId clientId) {
        List<AdGroupSimple> adGroups = mapList(frontpageWarningsGetItems, t -> toAdGroup(t, campaignId));
        Map<Long, Set<FrontpageCampaignShowType>> campaignShowTypeMap = (allowedFrontpageType == null) ? emptyMap() :
                ImmutableMap.of(campaignId, toFrontpageShowType(allowedFrontpageType));
        int shard = shardHelper.getShardByClientId(clientId);
        return cpmYndxFrontpageCurrencyService
                .getAdGroupIndexesToPriceDataMapByAdGroups(adGroups, shard, clientId, campaignShowTypeMap);
    }

    private static AdGroupSimple toAdGroup(FrontpageWarningsGetItem frontpageWarningsGetItem,
                                           Long campaignId) {
        return new AdGroup()
                .withCampaignId(campaignId)
                .withType(AdGroupType.CPM_YNDX_FRONTPAGE)
                .withGeo(nvl(convertGeo(frontpageWarningsGetItem.getGeo()), emptyList()));
    }

    private static AdGroupSimple toAdGroup(FrontpageGeoWarningsGetItem frontpageGeoWarningsGetItem,
                                           Long campaignId) {
        return new AdGroup()
                .withCampaignId(campaignId)
                .withType(AdGroupType.CPM_YNDX_FRONTPAGE)
                .withGeo(nvl(convertGeo(frontpageGeoWarningsGetItem.getGeo()), emptyList()));
    }

    private ValidationResult<FrontpageWarningsGetItem, Defect> validateWarningItem(
            CpmYndxFrontpageAdGroupPriceRestrictions adGroupRestriction, FrontpageWarningsGetItem item,
            Currency clientCurrency) {
        ItemValidationBuilder<FrontpageWarningsGetItem, Defect> validationResultBuilder =
                ItemValidationBuilder.of(item);
        validationResultBuilder.item(item.getPrice(), FrontpageWarningsGetItem.Prop.PRICE)
                .checkBy(new PriceValidator(clientCurrency, AdGroupType.CPM_YNDX_FRONTPAGE,
                        adGroupRestriction));
        return validationResultBuilder.getResult();
    }
}
