package ru.yandex.direct.web.entity.grid.controller;

import java.util.Collections;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import springfox.documentation.annotations.ApiIgnore;

import ru.yandex.direct.web.core.security.captcha.DisableAutoCaptcha;
import ru.yandex.direct.web.core.security.netacl.AllowNetworks;
import ru.yandex.direct.web.entity.grid.model.GridRequest;
import ru.yandex.direct.web.entity.grid.model.GridResponse;
import ru.yandex.direct.web.entity.grid.service.PublicGridGraphQLService;

import static ru.yandex.direct.common.net.NetworkName.INTERNAL;
import static ru.yandex.direct.web.core.security.configuration.BlackboxWebAuthenticationConfiguration.PUBLIC_BASE_URL;

/**
 * Публичный GraphQL-эндпоинт, не требуюющий аутентификации.
 * Аутентификация выключена, так как URL начинается с
 * {@link ru.yandex.direct.web.core.security.configuration.BlackboxWebAuthenticationConfiguration#PUBLIC_BASE_URL}.
 * <p>
 * Здесь производится только предварительная обработка запроса,
 * остальное делается в отдельном модуле libs-internal/grid-processing
 */
@ApiIgnore // не хотим показывать в swagger-е
@Controller
@ParametersAreNonnullByDefault
@RequestMapping(PUBLIC_BASE_URL + "grid")
public class PublicGraphQLController {
    private final PublicGridGraphQLService gridService;

    @Autowired
    public PublicGraphQLController(PublicGridGraphQLService gridService) {
        this.gridService = gridService;
    }

    /**
     * Статический интерфейс graphiql для отладки контроллера.
     * Доступен только из внутренней сети и только суперам и суперридерам
     */
    @GetMapping(value = "/")
    @DisableAutoCaptcha
    @AllowNetworks({INTERNAL})
    public String graphiqlInterface() {
        return "/static/graphiql.html";
    }

    @RequestMapping(
            value = "/api",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseBody
    @DisableAutoCaptcha
    public GridResponse executeGraphQL(@RequestBody GridRequest request) {
        Map<String, Object> variables = request.getVariables();
        if (variables == null) {
            variables = Collections.emptyMap();
        }
        return gridService.executeGraphQL(request.getOperationName(), request.getQuery(), variables);
    }
}
