package ru.yandex.direct.web.entity.grid.service;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import graphql.ExceptionWhileDataFetching;
import graphql.GraphQLError;
import graphql.GraphqlErrorHelper;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.CommonTranslations;
import ru.yandex.direct.core.TranslatableException;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.grid.processing.exception.GdExceptions;
import ru.yandex.direct.grid.processing.exception.GridRequestTimeoutException;
import ru.yandex.direct.grid.processing.exception.GridSkipInErrorsException;
import ru.yandex.direct.grid.processing.exception.GridValidationException;
import ru.yandex.direct.grid.processing.model.api.GdApiResponse;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.web.DirectWebTranslations;

import static java.lang.String.format;
import static ru.yandex.direct.libs.graphql.GraphqlHelper.REQ_ID_KEY;
import static ru.yandex.direct.libs.graphql.GraphqlHelper.addReqId;

/**
 * Сервис для обработки ошибок GraphQL сервисов
 */
@Service
@ParametersAreNonnullByDefault
public class GridErrorProcessingService {
    static final String VALIDATION_RESULT_FIELD = GdApiResponse.VALIDATION_RESULT.name();
    private static final String BETA_ERROR_DISCLAIMER =
            " (you're seeing this message because it is development environment)";
    private static final String MESSAGE_KEY = "message";
    private static final Logger logger = LoggerFactory.getLogger(GridErrorProcessingService.class);
    private static final String EMPTY_PATH = "";

    private final EnvironmentType environmentType;
    private final TranslationService translationService;

    public GridErrorProcessingService(EnvironmentType environmentType,
                                      TranslationService translationService) {
        this.environmentType = environmentType;
        this.translationService = translationService;
    }

    List<Object> sanitizeErrors(List<GraphQLError> graphQLErrors) {
        return StreamEx.of(graphQLErrors)
                .map(this::sanitizeError)
                .nonNull()
                .toList();
    }

    /**
     * Преобразование ошибки
     * Метод возвращает тип {@link GdValidationResult} для ошибок валидации или Map<String, Object> для остальных ошибок
     * Возвращает null, если ошибку не надо показывать
     */
    @Nullable
    private Object sanitizeError(GraphQLError error) {
        Map<String, Object> sanitaizedError = GraphqlErrorHelper.toSpecification(error);
        Object message = sanitaizedError.get(MESSAGE_KEY);
        if (error instanceof ExceptionWhileDataFetching) {
            Throwable cause = ((ExceptionWhileDataFetching) error).getException();
            if (cause instanceof InvocationTargetException) {
                cause = ((InvocationTargetException) cause).getTargetException();
            }

            if (cause instanceof GridValidationException) {
                GdValidationResult validationResult = ((GridValidationException) cause).getValidationResult();
                String graphQLPath = getGraphQLPath(error.getPath());
                return convertValidationResultToResponse(graphQLPath, validationResult);
            }

            if (cause instanceof GridSkipInErrorsException) {
                /*
                 * не отдаем фронту такие ошибки, иначе у них не будет доступа к data
                 * есть надежда, что в будущем фронт научится получать одновременно ошибки в errors и читать данные
                 * из data.
                 * не логируем, т.к. фронт для всех запросов будет запрашивать недоступные поля
                 */
                return null;
            }

            logger.error("Got exception during execution", cause);
            if (cause instanceof TranslatableException && containsPublicErrorCode(cause)) {
                //noinspection ConstantConditions
                sanitaizedError.put("code", GdExceptions.fromCode(((TranslatableException) cause).getCode()).getCode());
                message = translationService.translate(((TranslatableException) cause).getShortMessage());
            } else if (cause instanceof GridRequestTimeoutException) {
                message = translationService.translate(CommonTranslations.INSTANCE.timeLimitExceeded());
            } else if (environmentType.isBeta()) {
                // На бете тоже покажем сообщение, чтобы было проще отлаживаться
                message = cause.getMessage() + BETA_ERROR_DISCLAIMER;
            } else {
                message = translationService.translate(CommonTranslations.INSTANCE.serviceInternalError());
            }
        } else {
            logger.error("Got graphQLError: {}", error);
        }

        String supportMessage = translationService.translate(
                DirectWebTranslations.INSTANCE
                        .supportDataMessage(format("%s = %d", REQ_ID_KEY, Trace.current().getSpanId()))
        );

        sanitaizedError.put(MESSAGE_KEY, format("%s. %s", message, supportMessage));
        addReqId(sanitaizedError);
        return sanitaizedError;
    }

    /**
     * Преобразовывает список GraphQL путей в строку
     * Возвращает пустую строку, если получили ошибку
     */
    private static String getGraphQLPath(List<Object> graphQLPath) {
        try {
            return graphQLPath.stream()
                    .map(String.class::cast)
                    .collect(Collectors.joining("."));
        } catch (Exception e) {
            return EMPTY_PATH;
        }
    }

    private static boolean containsPublicErrorCode(Throwable cause) {
        return GdExceptions.fromCode(((TranslatableException) cause).getCode()) != null;
    }

    private static Map<String, Map<String, GdValidationResult>> convertValidationResultToResponse(String graphQLPath,
                                                                                                  GdValidationResult validationResult) {
        return ImmutableMap.of(graphQLPath, ImmutableMap.of(VALIDATION_RESULT_FIELD, validationResult));
    }
}
