package ru.yandex.direct.web.entity.grid.service;

import java.time.Instant;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import graphql.ExecutionResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.grid.processing.configuration.GridProcessingConfiguration;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.processor.GridGraphQLProcessor;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.grid.model.GridResponse;

import static ru.yandex.direct.grid.processing.configuration.GridProcessingConfiguration.PUBLIC_GRAPH_QL_PROCESSOR;
import static ru.yandex.direct.libs.graphql.GraphqlHelper.addReqId;

/**
 * Сервис для обработки публичных GraphQL-запросов, собирает контекст выполнения операции и передает запрос в
 * {@link GridProcessingConfiguration#PUBLIC_GRAPH_QL_PROCESSOR}
 * В контексте оператор либо {@link PublicGridGraphQLService#EMPTY_USER}
 * либо аутентифицированный пользователь паспорта, а subjectUser null
 */
@Service
@ParametersAreNonnullByDefault
public class PublicGridGraphQLService {
    private final GridGraphQLProcessor gridGraphQLProcessor;
    private final GridErrorProcessingService responseHandlingService;
    private final DirectWebAuthenticationSource authenticationSource;

    private static final User EMPTY_USER = new User().withRole(RbacRole.EMPTY).withUid(0L);

    @Autowired
    public PublicGridGraphQLService(@Qualifier(PUBLIC_GRAPH_QL_PROCESSOR) GridGraphQLProcessor gridGraphQLProcessor,
                                    GridErrorProcessingService responseHandlingService,
                                    DirectWebAuthenticationSource authenticationSource) {
        this.gridGraphQLProcessor = gridGraphQLProcessor;
        this.responseHandlingService = responseHandlingService;
        this.authenticationSource = authenticationSource;
    }

    public GridResponse executeGraphQL(String operationName, String query, Map<String, Object> arguments) {
        User operator = EMPTY_USER;
        if (authenticationSource.isAuthenticated()) {
            DirectAuthentication authentication = authenticationSource.getAuthentication();
            operator = authentication.getOperator();
        }

        ExecutionResult executionResult =
                gridGraphQLProcessor.processQuery(operationName, query, arguments,
                        new GridGraphQLContext(operator, null).withInstant(Instant.now()));

        addReqId(executionResult.getData());
        List<Object> errors = responseHandlingService.sanitizeErrors(executionResult.getErrors());

        return new GridResponse()
                .withData(executionResult.getData())
                .withErrors(errors)
                .withExtensions(executionResult.getExtensions());
    }
}
