package ru.yandex.direct.web.entity.internalads.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.rbac.model.ClientsRelationType;
import ru.yandex.direct.web.annotations.AllowedSubjectRoles;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.core.security.captcha.DisableAutoCaptcha;
import ru.yandex.direct.web.entity.internalads.model.CreateInternalAdProduct;
import ru.yandex.direct.web.entity.internalads.model.CreateInternalAdProductResponse;
import ru.yandex.direct.web.entity.internalads.model.GetProductListResponse;
import ru.yandex.direct.web.entity.internalads.model.UpdateInternalAdProduct;
import ru.yandex.direct.web.entity.internalads.model.UpdateInternalAdProductResponse;
import ru.yandex.direct.web.entity.internalads.service.InternalAdProductWebService;

import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_ADMIN;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_MANAGER;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;

@Controller
@RequestMapping(value = "/internalads/products", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "internalads-products")
@AllowedSubjectRoles({SUPER, INTERNAL_AD_ADMIN})
public class ProductsController {

    private final InternalAdProductWebService internalAdProductWebService;
    private final DirectWebAuthenticationSource directWebAuthenticationSource;


    @Autowired
    public ProductsController(InternalAdProductWebService internalAdProductWebService,
                              DirectWebAuthenticationSource directWebAuthenticationSource) {
        this.internalAdProductWebService = internalAdProductWebService;
        this.directWebAuthenticationSource = directWebAuthenticationSource;
    }

    @ApiOperation(
            value = "getProductList",
            httpMethod = "GET",
            nickname = "getProductList"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = GetProductListResponse.class)
            }
    )
    @RequestMapping(path = "/list", method = RequestMethod.GET)
    @AllowedSubjectRoles({SUPER, SUPERREADER, INTERNAL_AD_ADMIN, INTERNAL_AD_MANAGER, INTERNAL_AD_SUPERREADER})
    @ResponseBody
    public WebResponse getProducts() {
        User operator = directWebAuthenticationSource.getAuthentication().getOperator();
        switch (operator.getRole()) {
            case SUPER:
            case INTERNAL_AD_ADMIN:
                return internalAdProductWebService.getProductList(ClientsRelationType.INTERNAL_AD_PUBLISHER);

            case SUPERREADER:
            case INTERNAL_AD_SUPERREADER:
                return internalAdProductWebService.getProductList(ClientsRelationType.INTERNAL_AD_READER);

            case INTERNAL_AD_MANAGER:
                return internalAdProductWebService.getProductListWithAccess(operator.getClientId());

            default:
                throw new IllegalStateException("Unexpected operator role: " + operator.getRole());
        }
    }


    @ApiOperation(
            value = "createProduct",
            httpMethod = "POST",
            nickname = "createProduct"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = CreateInternalAdProductResponse.class)
            }
    )
    @PreAuthorizeWrite
    @DisableAutoCaptcha
    @RequestMapping(path = "/create", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse createProduct(@RequestBody CreateInternalAdProduct createInternalAdProduct) {
        return internalAdProductWebService.createProduct(createInternalAdProduct);
    }

    @ApiOperation(
            value = "updateProduct",
            httpMethod = "POST",
            nickname = "updateProduct"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = UpdateInternalAdProductResponse.class)
            }
    )
    @PreAuthorizeWrite
    @DisableAutoCaptcha
    @RequestMapping(path = "/update", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse updateProduct(@RequestBody UpdateInternalAdProduct updateInternalAdProduct) {
        return internalAdProductWebService.updateProduct(updateInternalAdProduct);
    }
}
