package ru.yandex.direct.web.entity.internalads.model;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.core.entity.client.model.ClientWithUsers;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsManagerProductAccess;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsManagerProductAccessType;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProduct;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductAccess;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductAccessType;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductOption;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductWithClient;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductWithClientAndAccess;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.rbac.model.ClientsRelationType;

import static com.google.common.base.Preconditions.checkNotNull;

public class InternalAdProductConverter {
    private InternalAdProductConverter() {
    }

    private static final Map<WebProductAccessType, InternalAdsProductAccessType> WEB_PRODUCT_ACCESS_CONVERTOR_MAP =
            ImmutableMap.<WebProductAccessType, InternalAdsProductAccessType>builder()
                    .put(WebProductAccessType.FULL, InternalAdsProductAccessType.INTERNAL_AD_PUBLISHER)
                    .put(WebProductAccessType.READONLY, InternalAdsProductAccessType.INTERNAL_AD_READER)
                    .build();

    private static final Map<ClientsRelationType, WebProductAccessType> CLIENTS_RELATION_TYPE_CONVERTOR_MAP =
            ImmutableMap.<ClientsRelationType, WebProductAccessType>builder()
                    .put(ClientsRelationType.INTERNAL_AD_PUBLISHER, WebProductAccessType.FULL)
                    .put(ClientsRelationType.INTERNAL_AD_READER, WebProductAccessType.READONLY)
                    .build();

    private static final Map<InternalAdsManagerProductAccessType, WebProductAccessType> ACCESS_MAP = Map.of(
            InternalAdsManagerProductAccessType.FULL, WebProductAccessType.FULL,
            InternalAdsManagerProductAccessType.READONLY, WebProductAccessType.READONLY,
            InternalAdsManagerProductAccessType.PARTIAL, WebProductAccessType.FULL);

    public static CreateInternalAdProductResponse toCreateInternalAdProductResponse(String login,
                                                                                    InternalAdsProduct product) {
        boolean isSoftware = product.getOptions().contains(InternalAdsProductOption.SOFTWARE);
        return new CreateInternalAdProductResponse()
                .withResult(new WebInternalAdProduct()
                        .withClientId(product.getClientId().asLong())
                        .withProductName(product.getName())
                        .withProductDescription(product.getDescription())
                        .withProductLogin(login)
                        .withProductAccess(ClientsRelationType.INTERNAL_AD_PUBLISHER)
                        .withProductIsSoftware(isSoftware)
                );
    }

    public static WebInternalAdProduct toWebInternalAdProduct(
            InternalAdsProductWithClient internalAdProduct,
            ClientsRelationType productAccess) {
        boolean isSoftware = internalAdProduct.getProduct().getOptions().contains(InternalAdsProductOption.SOFTWARE);
        return new WebInternalAdProduct()
                .withClientId(internalAdProduct.getClient().getClientId())
                .withProductName(internalAdProduct.getProduct().getName())
                .withProductDescription(internalAdProduct.getProduct().getDescription())
                .withProductLogin(getAndCheckSingleUser(internalAdProduct.getClient()).getLogin())
                .withProductAccess(productAccess)
                .withProductIsSoftware(isSoftware);
    }

    public static WebInternalAdProduct toWebInternalAdProduct(InternalAdsProductWithClientAndAccess internalAdProduct) {
        InternalAdsManagerProductAccess access = internalAdProduct.getAccess();
        final ClientsRelationType simulatedRelationType;
        final Set<Long> placeIds;
        switch (access.getAccessType()) {
            case FULL:
                simulatedRelationType = ClientsRelationType.INTERNAL_AD_PUBLISHER;
                placeIds = null;
                break;
            case PARTIAL:
                simulatedRelationType = ClientsRelationType.INTERNAL_AD_PUBLISHER;
                placeIds = access.getPlaceIds();
                break;
            case READONLY:
                simulatedRelationType = ClientsRelationType.INTERNAL_AD_READER;
                placeIds = null;
                break;

            default:
                throw new IllegalStateException("Unexpected value: " + access.getAccessType());
        }
        boolean isSoftware = internalAdProduct.getProduct().getOptions().contains(InternalAdsProductOption.SOFTWARE);
        return new WebInternalAdProduct()
                .withClientId(internalAdProduct.getClient().getClientId())
                .withProductName(internalAdProduct.getProduct().getName())
                .withProductDescription(internalAdProduct.getProduct().getDescription())
                .withProductLogin(getAndCheckSingleUser(internalAdProduct.getClient()).getLogin())
                .withProductAccess(simulatedRelationType)
                .withProductPlaceIds(placeIds)
                .withProductIsSoftware(isSoftware);
    }

    public static User getAndCheckSingleUser(ClientWithUsers clientWithUsers) {
        return clientWithUsers.getUsers().stream().findFirst()
                .orElseThrow(() -> new IllegalArgumentException("too many reps for product (expecting exactly one). " +
                        "Product clientId: " + clientWithUsers.getClientId()));
    }

    public static WebInternalAdUser toWebInternalAdUser(ClientWithUsers clientWithUsers,
                                                        List<WebInternalAdProductAccess> productsAccess) {
        User user = getAndCheckSingleUser(clientWithUsers);
        return new WebInternalAdUser()
                .withClientId(clientWithUsers.getClientId())
                .withLogin(user.getLogin())
                .withUsername(user.getFio())
                .withAccessibleProducts(productsAccess);
    }

    public static WebInternalAdProductAccess toWebInternalAdProductAccess(
            InternalAdsManagerProductAccess access,
            InternalAdsProductWithClient productWithClient) {
        InternalAdsProduct productInfo = productWithClient.getProduct();

        String productLogin = getAndCheckSingleUser(productWithClient.getClient()).getLogin();

        WebInternalAdProductAccess resultingAccess = new WebInternalAdProductAccess()
                .withProductClientId(access.getProductClientId().asLong())
                .withProductName(productInfo.getName())
                .withProductLogin(productLogin)
                .withAccessType(toWebProductAccessType(access.getAccessType()));

        if (access.getAccessType() == InternalAdsManagerProductAccessType.PARTIAL) {
            resultingAccess.setPlaceIds(checkNotNull(access.getPlaceIds()));
        }

        return resultingAccess;
    }

    private static WebProductAccessType toWebProductAccessType(InternalAdsManagerProductAccessType internalRelationType) {
        if (!ACCESS_MAP.containsKey(internalRelationType)) {
            throw new IllegalStateException("invalid value " + internalRelationType);
        }
        return ACCESS_MAP.get(internalRelationType);
    }

    public static InternalAdsManagerProductAccess toInternalAdsProductAccess(
            ClientId managerClientId, ModifyProductAccess modifyProductAccess) {
        Set<Long> placeIds = modifyProductAccess.getPlaceIds();
        final InternalAdsManagerProductAccessType accessType;
        final Set<Long> placeIdsToSave;
        switch (modifyProductAccess.getAccessType()) {
            case READONLY:
                accessType = InternalAdsManagerProductAccessType.READONLY;
                placeIdsToSave = null;
                break;
            case FULL:
                if (placeIds == null || placeIds.isEmpty()) {
                    accessType = InternalAdsManagerProductAccessType.FULL;
                    placeIdsToSave = null;
                } else {
                    accessType = InternalAdsManagerProductAccessType.PARTIAL;
                    placeIdsToSave = placeIds;
                }
                break;
            default:
                throw new IllegalStateException("Unexpected value: " + modifyProductAccess.getAccessType());
        }

        return InternalAdsManagerProductAccess.builder()
                .withManagerClientId(managerClientId)
                .withProductClientId(ClientId.fromLong(modifyProductAccess.getProductClientId()))
                .withAccessType(accessType)
                .withPlaceIds(placeIdsToSave)
                .build();
    }

    public static InternalAdsProductAccess toInternalAdsProductAccess(WebInternalAdProductAccess webInternalAdProductAccess) {

        return new InternalAdsProductAccess()
                .withProductId(ClientId.fromLong(webInternalAdProductAccess.getProductClientId()))
                .withAccessType(toInternalAccessType(webInternalAdProductAccess.getAccessType()))
                .withPermittedPlaces(webInternalAdProductAccess.getPlaceIds());
    }

    private static InternalAdsProductAccessType toInternalAccessType(WebProductAccessType webProductAccessType) {
        return WEB_PRODUCT_ACCESS_CONVERTOR_MAP.getOrDefault(webProductAccessType,
                InternalAdsProductAccessType.INTERNAL_AD_READER);
    }

}
