package ru.yandex.direct.web.entity.internalads.service;

import java.util.List;

import com.google.common.collect.ImmutableSet;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.internalads.service.InternalAdsProductService;
import ru.yandex.direct.core.entity.internalads.service.validation.defects.InternalAdDefects;
import ru.yandex.direct.core.entity.internalads.service.validation.defects.InternalAdsAccessGroupDefects;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.RbacClientsRelations;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.utils.TextConstants;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.AdmissibleCharsConstraint;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.internalads.model.CreateInternalAdProduct;
import ru.yandex.direct.web.entity.internalads.model.ModifyProductAccess;
import ru.yandex.direct.web.entity.internalads.model.RemoveProductAccess;
import ru.yandex.direct.web.entity.internalads.model.UpdateInternalAdProduct;

import static ru.yandex.direct.core.entity.internalads.service.validation.defects.InternalAdDefects.productWithThisNameAlreadyExists;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;

@Service
public class InternalAdProductValidationService {

    private final ClientService clientService;
    private final RbacClientsRelations rbacClientsRelations;
    private final ShardHelper shardHelper;
    private final InternalAdsProductService internalAdsProductService;

    private static final int MAX_NAME_LENGTH = 50;
    private static final int MAX_DESCRIPTION_LENGTH = 255;

    @Autowired
    public InternalAdProductValidationService(ClientService clientService,
                                              RbacClientsRelations rbacClientsRelations,
                                              ShardHelper shardHelper,
                                              InternalAdsProductService internalAdsProductService) {
        this.clientService = clientService;
        this.rbacClientsRelations = rbacClientsRelations;
        this.shardHelper = shardHelper;
        this.internalAdsProductService = internalAdsProductService;
    }

    private static final String PRODUCT_NAME_AVAILABLE_SYMBOLS = TextConstants.LAT_LETTERS + TextConstants.NUMBERS +
            "_-";

    public ValidationResult<CreateInternalAdProduct, Defect> validateCreateInternalAdProductRequest(CreateInternalAdProduct request) {
        ItemValidationBuilder<CreateInternalAdProduct, Defect> vb = ItemValidationBuilder.of(request);

        List<ClientId> productsWithSameName =
                internalAdsProductService.getClientIdsOfProductsByName(request.getProductName());

        vb.check(notNull());

        vb.item(request.getProductName(), CreateInternalAdProduct.PRODUCT_NAME)
                .check(productNameAdmissibleChars())
                .check(maxStringLength(MAX_NAME_LENGTH))
                .check(noProductsWithSameName(productsWithSameName));

        vb.item(request.getProductDescription(), CreateInternalAdProduct.PRODUCT_DESCRIPTION)
                .check(maxStringLength(MAX_DESCRIPTION_LENGTH));

        return vb.getResult();
    }

    public static Constraint<String, Defect> productNameAdmissibleChars() {
        return new AdmissibleCharsConstraint(PRODUCT_NAME_AVAILABLE_SYMBOLS);
    }

    private static Constraint<String, Defect> noProductsWithSameName(List<ClientId> clientIdsOfProductsWithSameName) {
        return Constraint.fromPredicate(
                productName -> clientIdsOfProductsWithSameName.isEmpty(),
                productWithThisNameAlreadyExists());
    }

    public ValidationResult<UpdateInternalAdProduct, Defect> validateUpdateInternalAdProductRequest(UpdateInternalAdProduct request) {
        ItemValidationBuilder<UpdateInternalAdProduct, Defect> vb = ItemValidationBuilder.of(request);

        vb.check(notNull());
        vb.item(request.getClientId(), UpdateInternalAdProduct.PRODUCT_ID)
                .check(CommonConstraints.validId());
        vb.item(request.getClientId(), UpdateInternalAdProduct.PRODUCT_ID)
                .check(Constraint.fromPredicate(shardHelper::isExistentClientId,
                        InternalAdDefects.unknownClientId()));
        vb.item(request.getProductDescription(), UpdateInternalAdProduct.PRODUCT_DESCRIPTION)
                .check(maxStringLength(MAX_DESCRIPTION_LENGTH));
        return vb.getResult();
    }

    public ValidationResult<ModifyProductAccess, Defect> validateUpdateInternalAdProductAccess(ClientId clientId,
                                                                                               ModifyProductAccess productToAccess) {
        ItemValidationBuilder<ModifyProductAccess, Defect> vb = ItemValidationBuilder.of(productToAccess);

        vb.check(notNull());
        vb.item(productToAccess.getProductClientId(), ModifyProductAccess.PRODUCT_ID)
                .check(CommonConstraints.validId())
                .check(Constraint.fromPredicate(shardHelper::isExistentClientId,
                        InternalAdDefects.unknownClientId()));
        vb.check(Constraint.fromPredicate(v -> checkProductRelationExists(clientId,
                ClientId.fromLong(v.getProductClientId())),
                InternalAdDefects.accessDoesNotExist()), When.isValid());

        return vb.getResult();
    }

    public ValidationResult<ModifyProductAccess, Defect> validateAddInternalAdProductAccess(ClientId clientId,
                                                                                            ModifyProductAccess productToAccess) {
        ItemValidationBuilder<ModifyProductAccess, Defect> vb = ItemValidationBuilder.of(productToAccess);

        vb.check(notNull());
        vb.item(productToAccess.getProductClientId(), ModifyProductAccess.PRODUCT_ID)
                .check(CommonConstraints.validId())
                .check(Constraint.fromPredicate(shardHelper::isExistentClientId,
                        InternalAdDefects.unknownClientId()));

        //todo проверить clientId на менеджера (DIRECT-100092)
        vb.check(Constraint.fromPredicate(v -> !checkProductRelationExists(clientId,
                ClientId.fromLong(v.getProductClientId())),
                InternalAdDefects.accessAlreadyExists()), When.isValid());

        return vb.getResult();
    }

    private boolean checkProductRelationExists(ClientId clientId, ClientId productId) {
        return rbacClientsRelations.isInternalAdRelation(clientId, productId);
    }


    public ValidationResult<List<RemoveProductAccess>, Defect> validateRemoveInternalAdProductAccess(ClientId clientId, List<RemoveProductAccess> request) {
        ListValidationBuilder<RemoveProductAccess, Defect> vb = ListValidationBuilder.of(request);

        vb.checkEach(notNull());
        vb.checkEach(Constraint.fromPredicate(r -> r.getProductClientId() != null, CommonDefects.notNull()));
        vb.checkEach(Constraint.fromPredicate(v -> checkProductRelationExists(clientId,
                ClientId.fromLong(v.getProductClientId())),
                InternalAdDefects.accessDoesNotExist()), When.isValid());

        return vb.getResult();
    }

    private Constraint<ClientId, Defect> clientIsInternalAdManager() {
        ImmutableSet<RbacRole> availableRoles = ImmutableSet.of(RbacRole.INTERNAL_AD_MANAGER);
        return Constraint.fromPredicate(clientId -> availableRoles.contains(clientService.getClient(clientId).getRole()),
                new Defect<>(InternalAdsAccessGroupDefects.Gen.INVALID_ROLE));
    }

}
