package ru.yandex.direct.web.entity.internalads.service;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.RandomStringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.client.model.ClientWithUsers;
import ru.yandex.direct.core.entity.client.service.AddClientService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProduct;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductOption;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductWithClient;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductWithClientAndAccess;
import ru.yandex.direct.core.entity.internalads.service.InternalAdsProductService;
import ru.yandex.direct.core.service.integration.passport.PassportService;
import ru.yandex.direct.core.service.integration.passport.RegisterUserRequest;
import ru.yandex.direct.core.service.integration.passport.RegisterUserResult;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.LoginOrUid;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.model.ClientsRelationType;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.entity.internalads.model.CreateInternalAdProduct;
import ru.yandex.direct.web.entity.internalads.model.GetProductListResponse;
import ru.yandex.direct.web.entity.internalads.model.InternalAdProductConverter;
import ru.yandex.direct.web.entity.internalads.model.UpdateInternalAdProduct;
import ru.yandex.direct.web.entity.internalads.model.UpdateInternalAdProductResponse;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.entity.internalads.model.InternalAdProductConverter.toCreateInternalAdProductResponse;
import static ru.yandex.direct.web.entity.internalads.model.InternalAdProductConverter.toWebInternalAdProduct;

@ParametersAreNonnullByDefault
@Service
public class InternalAdProductWebService {
    // паспорт требует имя и фамилию, чтобы создать учётную запись, а мы для внутренней рекламы
    // скрываем, что она вообще есть, кампании хранятся на "продукты", а что под ними клиенты,
    // особенность реализации; так что у всех учётных записей в паспорте под эти продукты
    // одни и те же имена и фамилии
    private static final String PASSPORT_FIRST_NAME = "Реклама";
    private static final String PASSPORT_LAST_NAME = "Внутренняя";

    private final InternalAdsProductService internalAdsProductService;
    private final InternalAdProductValidationService internalAdProductValidationService;
    private final ValidationResultConversionService validationResultConversionService;
    private final PassportService passportService;
    private final AddClientService addClientService;
    private final ClientService clientService;


    @Autowired
    public InternalAdProductWebService(InternalAdsProductService internalAdsProductService,
                                       InternalAdProductValidationService internalAdProductValidationService,
                                       ValidationResultConversionService validationResultConversionService,
                                       PassportService passportService,
                                       AddClientService addClientService,
                                       ClientService clientService) {
        this.internalAdsProductService = internalAdsProductService;
        this.internalAdProductValidationService = internalAdProductValidationService;
        this.validationResultConversionService = validationResultConversionService;
        this.passportService = passportService;
        this.addClientService = addClientService;
        this.clientService = clientService;
    }

    /**
     * Получить список всех продуктов
     */
    public GetProductListResponse getProductList(ClientsRelationType productAccess) {
        List<InternalAdsProductWithClient> productData = internalAdsProductService.getAllProducts();

        return new GetProductListResponse().withResult(
                mapList(productData, product -> toWebInternalAdProduct(product, productAccess))
        );
    }

    /**
     * Получить список все продуктов, доступных клиенту
     *
     * @param clientId - идентификатор клинета - маркетолога внутренней рекламы
     */
    public GetProductListResponse getProductListWithAccess(ClientId clientId) {
        List<InternalAdsProductWithClientAndAccess> productData =
                internalAdsProductService.getProductsAccessibleForClientId(clientId);

        return new GetProductListResponse().withResult(
                mapList(productData, InternalAdProductConverter::toWebInternalAdProduct)
        );
    }

    /**
     * Создать продукт внутренней рекламы
     */
    public WebResponse createProduct(CreateInternalAdProduct createInternalAdProduct) {

        ValidationResult<CreateInternalAdProduct, Defect> validationResult =
                internalAdProductValidationService.validateCreateInternalAdProductRequest(createInternalAdProduct);

        if (validationResult.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(validationResult);
        }

        String login = generatePassportLogin();

        RegisterUserResult passportRegisterUserResult = createPassportUserForInternalAdProduct(login);
        if (!passportRegisterUserResult.getStatus().isSuccess()) {
            return new WebErrorResponse("failed to register passport user: " + passportRegisterUserResult.getStatus().name(),
                    passportRegisterUserResult.getStatus().toString());
        }

        Result<UidAndClientId> directAddClientResult = addClientService.processRequest(LoginOrUid.of(login),
                PASSPORT_FIRST_NAME + " " + PASSPORT_LAST_NAME,
                Region.RUSSIA_REGION_ID, CurrencyCode.RUB, RbacRole.CLIENT);

        if (!directAddClientResult.isSuccessful()) {
            return new WebErrorResponse("Error while creating direct client",
                    directAddClientResult.getErrors().toString());
        }

        ClientId clientId = directAddClientResult.getResult().getClientId();
        clientService.setClientInternalAdProductPerm(clientId);

        InternalAdsProduct product = new InternalAdsProduct()
                .withClientId(clientId)
                .withName(createInternalAdProduct.productName)
                .withDescription(createInternalAdProduct.productDescription)
                .withOptions(toInternalAdsProductOptions(createInternalAdProduct.getProductIsSoftware()));

        if (!internalAdsProductService.createProduct(product)) {
            return new WebErrorResponse("Error while creating internal ad product",
                    "product clientId " + clientId);
        }

        return toCreateInternalAdProductResponse(login, product);
    }

    private String generatePassportLogin() {
        return "yndx-direct-iap-" + RandomStringUtils.randomAlphabetic(5).toLowerCase();
    }

    private RegisterUserResult createPassportUserForInternalAdProduct(String login) {
        String requestId = Long.toString(Trace.current().getTraceId());
        String remoteIp = HttpUtil.getRemoteAddress().orElseThrow(IllegalStateException::new).getHostAddress();
        RegisterUserRequest registerUserRequest = new RegisterUserRequest()
                .withLogin(login)
                .withFirstName(PASSPORT_FIRST_NAME)
                .withLastName(PASSPORT_LAST_NAME);

        return passportService.registerUser(requestId, registerUserRequest, remoteIp);
    }

    /**
     * Обновление данных о продукте внутренней рекламы
     */
    public WebResponse updateProduct(UpdateInternalAdProduct updateInternalAdProduct) {
        ValidationResult<UpdateInternalAdProduct, Defect> validationResult =
                internalAdProductValidationService.validateUpdateInternalAdProductRequest(updateInternalAdProduct);

        if (validationResult.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(validationResult);
        }

        ClientId productClientId = ClientId.fromLong(updateInternalAdProduct.getClientId());
        internalAdsProductService.updateProduct(new InternalAdsProduct()
                .withClientId(productClientId)
                .withDescription(updateInternalAdProduct.getProductDescription())
                .withOptions(toInternalAdsProductOptions(updateInternalAdProduct.getProductIsSoftware())));

        InternalAdsProduct updatedProduct = internalAdsProductService.getProduct(productClientId);

        ClientWithUsers client = clientService.getClientWithUsers(productClientId);
        InternalAdsProductWithClient productWithClient = InternalAdsProductWithClient.of(updatedProduct, client);

        return new UpdateInternalAdProductResponse()
                .withResult(toWebInternalAdProduct(productWithClient, ClientsRelationType.INTERNAL_AD_PUBLISHER));
    }

    private static Set<InternalAdsProductOption> toInternalAdsProductOptions(@Nullable Boolean productIsSoftware) {
        return nvl(productIsSoftware, false)
                ? Set.of(InternalAdsProductOption.SOFTWARE)
                : Collections.emptySet();
    }

}
