package ru.yandex.direct.web.entity.internaltools.controller;

import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.view.RedirectView;
import springfox.documentation.annotations.ApiIgnore;

import ru.yandex.direct.core.entity.trustedredirects.model.Opts;
import ru.yandex.direct.core.entity.trustedredirects.model.RedirectType;
import ru.yandex.direct.core.entity.trustedredirects.model.TrustedRedirects;
import ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsService;
import ru.yandex.direct.dbschema.ppcdict.enums.TrustedRedirectsRedirectType;
import ru.yandex.direct.web.annotations.AllowedOperatorRoles;
import ru.yandex.direct.web.annotations.AllowedSubjectRoles;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.csrf.CsrfCheck;
import ru.yandex.direct.web.core.security.netacl.AllowNetworks;

import static ru.yandex.direct.common.net.NetworkName.INTERNAL;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;

@ApiIgnore // не хотим показывать в swagger-е
@Controller
@ParametersAreNonnullByDefault
@RequestMapping(value = "/trusted_redirects", produces = MediaType.TEXT_HTML_VALUE)
@AllowNetworks({INTERNAL})
@AllowedOperatorRoles({SUPER, SUPPORT})
@AllowedSubjectRoles({SUPER, SUPPORT})
public class TrustedRedirectsController {

    private static final Logger logger = LoggerFactory.getLogger(TrustedRedirectsController.class);

    private final TrustedRedirectsService trustedRedirectsService;

    @Autowired
    public TrustedRedirectsController(TrustedRedirectsService trustedRedirectsService) {
        this.trustedRedirectsService = trustedRedirectsService;
    }

    @RequestMapping(
            path = "/add",
            method = RequestMethod.POST
    )
    @CsrfCheck(enabled = false)
    public RedirectView add(
            @RequestParam(value = "new_host") String href,
            @RequestParam(value = "redirect_type") String stringRedirectType,
            @RequestParam(value = "opts_https_only", required = false) String optHttpsOnly,
            @RequestParam(value = "opts_allow_wildcard", required = false) String optWildcardAllowed,
            HttpServletRequest request) {
        addInternal(href, stringRedirectType, optHttpsOnly, optWildcardAllowed);
        String referrer = getReferrer(request);
        return new RedirectView(referrer);
    }

    private boolean addInternal(String href, String stringRedirectType, String optHttpsOnly,
                                String optWildcardAllowed) {
        TrustedRedirectsRedirectType redirectType = redirectTypeFromString(stringRedirectType);
        TrustedRedirects trustedRedirects = new TrustedRedirects();
        trustedRedirects.withDomain(href);
        trustedRedirects.withRedirectType(RedirectType.fromSource(redirectType));
        Set<Opts> opts = new HashSet<>();
        if ("on".equals(optHttpsOnly)) {
            opts.add(Opts.https_only);
        }
        if ("on".equals(optWildcardAllowed)) {
            opts.add(Opts.allow_wildcard);
        }
        trustedRedirects.withOpts(opts);
        boolean isAdded = trustedRedirectsService.addTrustedDomain(trustedRedirects);
        if (!isAdded) {
            logger.info("Domain not added " + trustedRedirects.getDomain());
        }
        return isAdded;
    }

    @RequestMapping(path = "/add-ajax", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public WebResponse addAjax(
            @RequestParam(value = "new_host") String href,
            @RequestParam(value = "redirect_type") String stringRedirectType,
            @RequestParam(value = "opts_https_only", required = false) String optHttpsOnly,
            @RequestParam(value = "opts_allow_wildcard", required = false) String optWildcardAllowed) {

        boolean isAdded = addInternal(href, stringRedirectType, optHttpsOnly, optWildcardAllowed);
        if (isAdded) {
            return new WebSuccessResponse();
        } else {
            return new WebErrorResponse("Invalid domain", "Invalid domain");
        }
    }

    @RequestMapping(
            path = "/delete",
            method = RequestMethod.POST
    )
    @CsrfCheck(enabled = false)
    public RedirectView delete(
            @RequestParam(value = "rm") String domain,
            @RequestParam(value = "redirect_type") String stringRedirectType,
            HttpServletRequest request) {

        TrustedRedirectsRedirectType redirectType = redirectTypeFromString(stringRedirectType);
        TrustedRedirects trustedRedirects = new TrustedRedirects();
        trustedRedirects.withDomain(domain);
        trustedRedirects.withRedirectType(RedirectType.fromSource(redirectType));

        boolean result = trustedRedirectsService.deleteDomain(trustedRedirects);
        if (!result) {
            logger.warn("Domain not deleted " + trustedRedirects.getDomain());
        }
        String referrer = getReferrer(request);
        return new RedirectView(referrer);
    }

    private static String getReferrer(HttpServletRequest request) {
        String referrer = request.getHeader("referer");
        if (referrer == null || referrer.isEmpty()) {
            return "/";
        }
        // prevent infinite loop
        if (referrer.contains("/trusted_redirects")) {
            return "/";
        }
        return referrer;
    }

    private static TrustedRedirectsRedirectType redirectTypeFromString(String redirectType) {
        Set<TrustedRedirectsRedirectType> allTypes = EnumSet.allOf(TrustedRedirectsRedirectType.class);
        for (TrustedRedirectsRedirectType type : allTypes) {
            if (type.getLiteral().equals(redirectType)) {
                return type;
            }
        }
        return null;
    }
}
