package ru.yandex.direct.web.entity.internaltools.service;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.core.InternalToolProxy;
import ru.yandex.direct.internaltools.core.InternalToolsRegistry;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolAccessDeniedException;
import ru.yandex.direct.internaltools.core.exception.InternalToolProcessingException;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.input.InternalToolInput;
import ru.yandex.direct.internaltools.core.input.InternalToolInputGroup;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.internaltools.model.InternalToolBasicDescription;
import ru.yandex.direct.web.entity.internaltools.model.InternalToolCategoryDescription;
import ru.yandex.direct.web.entity.internaltools.model.InternalToolExtendedDescription;
import ru.yandex.direct.web.entity.internaltools.model.InternalToolInputGroupRepresentation;
import ru.yandex.direct.web.entity.internaltools.model.InternalToolInputRepresentation;
import ru.yandex.direct.web.entity.internaltools.model.InternalToolsFileDescription;

import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE_CACHE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class InternalToolsService {
    private static final Logger logger = LoggerFactory.getLogger(InternalToolsService.class);

    private static final long MAX_UPLOAD_FILE_SIZE = 1024 * 1024 * 50L;
    private static final Comparator<InternalToolBasicDescription> TOOL_DESCRIPTION_COMPARATOR =
            Comparator.comparing(InternalToolBasicDescription::getName);
    static final Duration MAX_FILE_LIFE_DURATION = Duration.ofMinutes(5);

    private final InternalToolsRegistry toolsRegistry;
    private final DirectWebAuthenticationSource authenticationSource;
    private final LettuceConnectionProvider lettuceCache;

    public InternalToolsService(InternalToolsRegistry toolsRegistry,
                                DirectWebAuthenticationSource authenticationSource,
                                @Qualifier(LETTUCE_CACHE) LettuceConnectionProvider lettuceCache) {
        this.toolsRegistry = toolsRegistry;
        this.authenticationSource = authenticationSource;
        this.lettuceCache = lettuceCache;
    }

    /**
     * Получить set с ролями переданного пользователя во внутренних инструментах
     */
    private Set<InternalToolAccessRole> getUserRoles(User operator) {
        RbacRole role = operator.getRole();
        Set<InternalToolAccessRole> roles = EnumSet.noneOf(InternalToolAccessRole.class);
        // TODO(bzzzz): роли SPECIFIC_SUPER_USER, LOG_READER и CATALOG_MODERATOR пока не поддерживаем
        if (role == RbacRole.SUPER) {
            roles.add(InternalToolAccessRole.SUPER);
        } else if (role == RbacRole.SUPERREADER) {
            if (operator.getDeveloper()) {
                roles.add(InternalToolAccessRole.DEVELOPER);
            }
            roles.add(InternalToolAccessRole.SUPERREADER);
        } else if (role == RbacRole.SUPPORT) {
            roles.add(InternalToolAccessRole.SUPPORT);
        } else if (role == RbacRole.PLACER) {
            roles.add(InternalToolAccessRole.PLACER);
        } else if (role == RbacRole.MEDIA) {
            roles.add(InternalToolAccessRole.MEDIAPLANNER);
        } else if (role == RbacRole.MANAGER) {
            roles.add(InternalToolAccessRole.MANAGER);
        } else if (role == RbacRole.INTERNAL_AD_ADMIN) {
            roles.add(InternalToolAccessRole.INTERNAL_AD_ADMIN);
        } else {
            roles.add(InternalToolAccessRole.INTERNAL_USER);
        }
        return roles;
    }

    /**
     * Получить список категорий внутренних инструментов и их доступного оператору содержимого.
     * Если в категориях нет ничего, доступного оператору, они удаляются из ответа
     */
    public List<InternalToolCategoryDescription> getToolsCategories() {
        User operator = authenticationSource.getAuthentication().getOperator();
        ClientId operatorClientId = operator.getClientId();
        Set<InternalToolAccessRole> userRoles = getUserRoles(operator);
        logger.debug("User {} with rights {} tried to get tools list", operator.getLogin(), userRoles);

        Map<InternalToolCategory, List<InternalToolProxy>> internalToolsByCategory =
                toolsRegistry.getInternalToolsByCategory(userRoles, operatorClientId);
        List<InternalToolCategoryDescription> result = new ArrayList<>();
        for (InternalToolCategory category : InternalToolCategory.values()) {
            if (!internalToolsByCategory.containsKey(category)) {
                continue;
            }
            List<InternalToolProxy> internalToolProxies = internalToolsByCategory.get(category);
            List<InternalToolBasicDescription> descriptions = extractOrderedDescriptions(internalToolProxies);
            result.add(new InternalToolCategoryDescription()
                    .withName(category.getName())
                    .withItems(descriptions));
        }

        return result;
    }

    /**
     * Из переданной коллекции {@code internalToolProxies} метод извлекает описания {@link InternalToolBasicDescription}
     * и возвращает их список, упорядоченный по имени (см. {@link #TOOL_DESCRIPTION_COMPARATOR})
     */
    static List<InternalToolBasicDescription> extractOrderedDescriptions(
            Collection<InternalToolProxy> internalToolProxies) {
        return internalToolProxies.stream()
                .map(InternalToolsService::generateBasicDescription)
                .sorted(TOOL_DESCRIPTION_COMPARATOR)
                .collect(Collectors.toList());
    }

    private static InternalToolBasicDescription generateBasicDescription(InternalToolProxy proxy) {
        return new InternalToolBasicDescription()
                .withName(proxy.getName())
                .withLabel(proxy.getLabel())
                .withDescription(proxy.getDescription());
    }

    /**
     * Получить полное описание внутреннего инструмента, если он доступен оператору
     */
    public InternalToolExtendedDescription getToolDescription(String label) {
        User operator = authenticationSource.getAuthentication().getOperator();
        Set<InternalToolAccessRole> userRoles = getUserRoles(operator);
        logger.debug("User {} with rights {} tried to get tool {} description", operator.getLogin(), userRoles, label);

        InternalToolProxy<?> proxy = toolsRegistry.getInternalToolProxy(label, userRoles, operator.getClientId());
        return new InternalToolExtendedDescription()
                .withAction(proxy.getAction().getName())
                .withMethod(proxy.getType() == InternalToolType.REPORT ? HttpMethod.GET : HttpMethod.POST)
                .withDisclaimers(proxy.getDisclaimers())
                .withInputGroups(mapList(proxy.getInputGroups(), this::generateInputGroupDescription))
                .withBareRunData(proxy.bareRun())
                .withName(proxy.getName())
                .withLabel(proxy.getLabel())
                .withDescription(proxy.getDescription());
    }

    private InternalToolInputGroupRepresentation generateInputGroupDescription(InternalToolInputGroup<?> group) {
        return new InternalToolInputGroupRepresentation()
                .withName(group.getName())
                .withInputs(mapList(group.getInputList(), this::generateInputDescription));
    }

    private InternalToolInputRepresentation generateInputDescription(InternalToolInput<?, ?> input) {
        return new InternalToolInputRepresentation()
                .withName(input.getName())
                .withLabel(input.getLabel())
                .withDescription(input.getDescription())
                .withInputType(input.getInputType().name().toLowerCase())
                .withDefaultValue(input.getDefaultValue())
                .withArgs(input.getArgs())
                .withAllowedValues(input.getAllowedValues())
                .withRequired(input.isRequired());
    }

    /**
     * Получить результат работы внутреннего инструмента, выполняющего только читающие запросы к базе
     */
    public InternalToolResult processReadingTool(String label, Map<String, Object> webRequest) {
        User operator = authenticationSource.getAuthentication().getOperator();
        Set<InternalToolAccessRole> userRoles = getUserRoles(operator);
        logger.debug("User {} with rights {} tried to use GET tool {}", operator.getLogin(), userRoles, label);

        InternalToolProxy<?> proxy = toolsRegistry.getInternalToolProxy(label, userRoles, operator.getClientId());
        if (proxy.writesData()) {
            throw new InternalToolAccessDeniedException(String.format("Wrong request method for tool %s", label));
        }
        return processTool(proxy, webRequest, operator);
    }

    /**
     * Получить результат работы внутреннего инструмента, изменяющего объекты в Директе
     */
    public InternalToolResult processWritingTool(String label, Map<String, Object> webRequest) {
        User operator = authenticationSource.getAuthentication().getOperator();
        Set<InternalToolAccessRole> userRoles = getUserRoles(operator);
        logger.debug("User {} with rights {} tried to use POST tool {}", operator.getLogin(), userRoles, label);

        InternalToolProxy<?> proxy = toolsRegistry.getInternalToolProxy(label, userRoles, operator.getClientId());
        if (!proxy.writesData()) {
            throw new InternalToolAccessDeniedException(String.format("Wrong request method for tool %s", label));
        }
        return processTool(proxy, webRequest, operator);
    }

    private InternalToolResult processTool(InternalToolProxy<?> proxy, Map<String, Object> webRequest, User operator) {
        Map<String, Object> modifiedRequest = new HashMap<>(webRequest);
        if (proxy.isAcceptsFiles()) {
            Set<String> fileFields = proxy.getFileFieldsNames();
            for (String fileField : fileFields) {
                String fieldValue = (String) modifiedRequest.get(fileField);
                if (fieldValue != null) {
                    modifiedRequest.put(fileField, getFileContents(fieldValue, operator));
                }
            }
        }
        return proxy.process(modifiedRequest, operator);
    }

    /**
     * Сохранить файл в кеше для того, чтобы он мог быть использован в работе инструмента
     */
    public InternalToolsFileDescription saveFile(String label, MultipartFile file) {
        User operator = authenticationSource.getAuthentication().getOperator();
        InternalToolProxy<?> proxy = toolsRegistry.getInternalToolProxy(label, getUserRoles(operator),
                operator.getClientId());
        if (!proxy.isAcceptsFiles()) {
            throw new InternalToolProcessingException(String.format("Tool %s does not need this file", label));
        } else if (file.getSize() > MAX_UPLOAD_FILE_SIZE) {
            throw new InternalToolValidationException("File is too big");
        }
        String fileKey = generateFileKey(label);
        try {
            byte[] fileContent = file.getBytes();
            lettuceCache.callBinary("redis:setex", cmd ->
                    cmd.setex(fileKey.getBytes(), MAX_FILE_LIFE_DURATION.getSeconds(), fileContent)
            );
            var fileOwnerKey = generateFileOwnerKey(fileKey);
            logger.info("File owner key for upload {}", fileOwnerKey);
            lettuceCache.call("redis:setex", cmd ->
                    cmd.setex(
                            fileOwnerKey,
                            MAX_FILE_LIFE_DURATION.getSeconds(),
                            operator.getUid().toString())
            );
        } catch (IOException e) {
            throw new InternalToolProcessingException("Error when saving file", e);
        }
        return new InternalToolsFileDescription().withKey(fileKey);
    }

    private String generateFileKey(String label) {
        return String.format("ITF/%s/%s", label, UUID.randomUUID().toString());
    }

    private String generateFileOwnerKey(String fileKey) {
        return String.format("%s/owner", fileKey);
    }

    @Nullable
    private byte[] getFileContents(String fieldValue, User operator) {
        var fileOwnerKey = generateFileOwnerKey(fieldValue);
        logger.info("Field value = {}, file owner key {}", fieldValue, fileOwnerKey);
        String fileOwnerUserId = lettuceCache.call("redis:get", cmd -> cmd.get(fileOwnerKey));
        if (fileOwnerUserId == null) {
            return null;
        }
        if (!fileOwnerUserId.equals(operator.getUid().toString())) {
            throw new InternalToolAccessDeniedException("You are trying to access file you do not own");
        }
        return lettuceCache.callBinary("redis:get", cmd -> cmd.get(fieldValue.getBytes()));
    }
}
