package ru.yandex.direct.web.entity.inventori.controller;

import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import com.fasterxml.jackson.core.JsonProcessingException;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.inventori.InventoriException;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.entity.inventori.model.CpmForecastRequest;
import ru.yandex.direct.web.core.entity.inventori.model.CpmForecastResult;
import ru.yandex.direct.web.core.entity.inventori.model.CpmTrafficLightPredictionResult;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCpmRecommendationRequest;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCpmRecommendationResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachIndoorRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachIndoorResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachInfo;
import ru.yandex.direct.web.core.entity.inventori.model.ReachOutdoorRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachOutdoorResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachRecommendationResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachResult;
import ru.yandex.direct.web.core.entity.inventori.service.InventoriWebService;
import ru.yandex.direct.web.core.entity.inventori.service.InventoriWebValidationService;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.core.security.captcha.DisableAutoCaptcha;
import ru.yandex.direct.web.entity.inventori.model.CpmForecastResponse;
import ru.yandex.direct.web.entity.inventori.model.CpmTrafficLightPredictionResponse;
import ru.yandex.direct.web.entity.inventori.model.GeneralRecommendationResponse;
import ru.yandex.direct.web.entity.inventori.model.ReachIndoorResponse;
import ru.yandex.direct.web.entity.inventori.model.ReachOutdoorResponse;
import ru.yandex.direct.web.entity.inventori.model.ReachRecommendationResponse;
import ru.yandex.direct.web.entity.inventori.model.ReachResponse;
import ru.yandex.direct.web.entity.inventori.service.CampaignForecastService;
import ru.yandex.direct.web.entity.inventori.service.CampaignForecastValidationService;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;

@Controller
@RequestMapping(value = "/mediareach",
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "mediareach")
public class InventoriController {

    private static final Logger log = LoggerFactory.getLogger(InventoriController.class);
    private static final double CPV_RED_UPPER_LIMIT_RATIO = 0.15;
    private static final double CPV_YELLOW_UPPER_LIMIT_RATIO = 0.25;
    private static final Set<String>
            FAKE_FORECAST_STRATEGIES = Set.of("AVG_CPV", "MAX_AVG_CPV", "MAX_AVG_CPV_CUSTOM_PERIOD");

    private final TranslationService translationService;
    private final InventoriWebService inventoriWebService;
    private final InventoriWebValidationService inventoriValidationService;
    private final ValidationResultConversionService validationResultConversionService;
    private final CampaignForecastValidationService campaignForecastValidationService;
    private final CampaignForecastService campaignForecastService;
    private final DirectWebAuthenticationSource authSource;
    private final ClientService clientService;

    @Autowired
    public InventoriController(
            TranslationService translationService,
            InventoriWebService inventoriWebService,
            InventoriWebValidationService inventoriValidationService,
            ValidationResultConversionService validationResultConversionService,
            DirectWebAuthenticationSource authSource,
            CampaignForecastValidationService campaignForecastValidationService,
            CampaignForecastService campaignForecastService,
            ClientService clientService) {
        this.translationService = translationService;
        this.inventoriWebService = inventoriWebService;
        this.inventoriValidationService = inventoriValidationService;
        this.validationResultConversionService = validationResultConversionService;
        this.campaignForecastValidationService = campaignForecastValidationService;
        this.campaignForecastService = campaignForecastService;
        this.authSource = authSource;
        this.clientService = clientService;
    }

    @ExceptionHandler(InventoriException.class)
    public ResponseEntity<WebResponse> handleInventoriClientException(InventoriException e) {
        log.error("Handling InventoriException", e);
        return createServiceUnavailableWebResponse(e);
    }

    private ResponseEntity<WebResponse> createServiceUnavailableWebResponse(
            InventoriException e) {
        return new ResponseEntity<>(new InventoriWebErrorResponse(HttpStatus.GATEWAY_TIMEOUT.value(),
                translationService.translate(
                        InventoriTranslations.INSTANCE.forecastServiceUnavailable())).withRequestId(e.getRequestId()),
                HttpStatus.GATEWAY_TIMEOUT);
    }

    /**
     * Ручка расчета охвата аудитории группы объявлений с учетом выбранных аудиторных условий
     *
     * @param request      <a href="https://wiki.yandex-team.ru/users/aliho/projects/direct/cryptainventory/#zapros">запрос</a>
     * @param subjectLogin - логин пользователя
     * @return <a href="https://wiki.yandex-team.ru/users/aliho/projects/direct/cryptainventory/#otvet">ответ</a>
     * @see <a href="https://wiki.yandex-team.ru/users/aliho/projects/direct/cryptainventory/#post/web-api/mediareach/reach">WIKI</a>
     */
    @ApiOperation(
            value = "getReach",
            httpMethod = "POST",
            nickname = "getReach"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 500, message = "Internal server error", response = WebErrorResponse.class),
                    @ApiResponse(code = 501, message = "Not implemented", response = WebErrorResponse.class),
                    @ApiResponse(code = 504, message = "Gateway Timeout", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = ReachResponse.class)
            }
    )

    @RequestMapping(path = "/reach",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public ResponseEntity<WebResponse> getReach(@RequestBody ReachRequest request,
                                                @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN) String subjectLogin) {
        ValidationResult<ReachRequest, Defect> preValidationResult =
                inventoriValidationService.validate(request);

        if (preValidationResult.hasAnyErrors()) {
            return new ResponseEntity<>(validationResultConversionService.buildValidationResponse(preValidationResult),
                    HttpStatus.BAD_REQUEST);
        }

        ReachResult result = inventoriWebService.getReachForecast(request);

        log.debug("Response: {}", JsonUtils.toJson(result));

        return new ResponseEntity<>(new ReachResponse(result), Optional.of(result).map(
                ReachResult::getDetailed).map(ReachInfo::getErrors).isPresent() ? HttpStatus.NOT_IMPLEMENTED
                : HttpStatus.OK);
    }

    @ApiOperation(
            value = "getReachOutdoor",
            httpMethod = "POST",
            nickname = "getReachOutdoor",
            notes = "Ручка расчета охвата аудитории outdoor группы объявлений"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = ReachResponse.class)
            }
    )

    @RequestMapping(path = "/reach_outdoor",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public ResponseEntity<WebResponse> getReachOutdoor(@RequestBody ReachOutdoorRequest request,
                                                       @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN) String subjectLogin) {
        DirectAuthentication auth = authSource.getAuthentication();
        ClientId clientId = auth.getSubjectUser().getClientId();

        ValidationResult<ReachOutdoorRequest, Defect> preValidationResult =
                inventoriValidationService.validate(request, clientId);

        if (preValidationResult.hasAnyErrors()) {
            return new ResponseEntity<>(validationResultConversionService.buildValidationResponse(preValidationResult),
                    HttpStatus.BAD_REQUEST);
        }

        ReachOutdoorResult result = inventoriWebService.getReachOutdoorForecast(request);

        return new ResponseEntity<>(new ReachOutdoorResponse(result), HttpStatus.OK);
    }

    @ApiOperation(
            value = "getReachIndoor",
            httpMethod = "POST",
            nickname = "getReachIndoor",
            notes = "Ручка расчета охвата аудитории indoor группы объявлений"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = ReachResponse.class)
            }
    )

    @RequestMapping(path = "/reach_indoor",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public ResponseEntity<WebResponse> getReachIndoor(@RequestBody ReachIndoorRequest request,
                                                      @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN) String subjectLogin) {
        DirectAuthentication auth = authSource.getAuthentication();
        ClientId clientId = auth.getSubjectUser().getClientId();

        ValidationResult<ReachIndoorRequest, Defect> preValidationResult =
                inventoriValidationService.validate(request, clientId);

        if (preValidationResult.hasAnyErrors()) {
            return new ResponseEntity<>(validationResultConversionService.buildValidationResponse(preValidationResult),
                    HttpStatus.BAD_REQUEST);
        }

        ReachIndoorResult result = inventoriWebService.getReachIndoorForecast(request);

        return new ResponseEntity<>(new ReachIndoorResponse(result), HttpStatus.OK);
    }

    /**
     * <p>Метод расчета рекомендаций для охвата аудитории группы объявлений с учетом выбранных аудиторных условий.</p>
     * <p>Особенности аналогичные как и у метода
     * <a href="https://wiki.yandex-team.ru/users/aliho/projects/direct/cryptainventory/#osobennosti">/reach/</a></p>
     *
     * <p>Основное отличие:</p>
     * <p>1. Сделать отдельный запрос за охватом (аналогично POST /web-api/inventori/reach/) и сохранив от него result.detailed.reach,
     * назовем A.</p>
     * <p>2. Сделать выборку из доступных типо-размеров (РАСПИСАТЬ ОТКУДА ИХ ВЗЯТЬ) для объявлений (баннеров) = из всех доступных вычесть те, что
     * передали с фронта в параметре banner_creatives.</p>
     * <p>3. Для каждого элемента полученного массива выполнить запрос в Inventori за охватом, добавляя его к переданному со фронта массиву
     * banner_creatives .</p>
     * <p>4. Для каждого запроса получить охват с учетом крипты  result.detailed.reach , назовем  B1...Bn</p>
     * <p>5. Для каждого запроса рассчитать рекомендацию по формуле:  C = Bn - A / 100 * A и отдать его в поле increase_percent</p>
     *
     * @param request      <a href="https://wiki.yandex-team.ru/users/aliho/projects/direct/cryptainventory/#zapros">запрос</a>
     * @param subjectLogin - логин пользователя
     * @return <a href="https://wiki.yandex-team.ru/users/aliho/projects/direct/cryptainventory/#otvet1">ответ</a>
     * @see <a href="https://wiki.yandex-team.ru/users/aliho/projects/direct/cryptainventory/#post/web-api/mediareach/reach/recommendation">WIKI</a>
     */
    @ApiOperation(
            value = "getReachRecommendation",
            httpMethod = "POST",
            nickname = "getReachRecommendation"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 501, message = "Not implemented", response = WebErrorResponse.class),
                    @ApiResponse(code = 504, message = "Gateway Timeout", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = ReachRecommendationResponse.class)
            }
    )

    @RequestMapping(path = "/reach/recommendation",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public ResponseEntity<WebResponse> getReachRecommendation(@RequestBody ReachRequest request,
                                                              @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN) String subjectLogin) {

        ReachRecommendationResult result = inventoriWebService.getReachRecommendation(request);

        log.debug("Response: {}", JsonUtils.toJson(result));

        return new ResponseEntity<>(new ReachRecommendationResponse(result), HttpStatus.OK);
    }

    @ApiOperation(
            value = "getCampaignForecast",
            httpMethod = "POST",
            nickname = "getCampaignForecast"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 200, message = "Ok", response = CpmForecastResponse.class),
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 501, message = "Not implemented", response = WebErrorResponse.class),
                    @ApiResponse(code = 504, message = "Gateway Timeout", response = WebErrorResponse.class),
            }
    )
    @PreAuthorizeRead
    @DisableAutoCaptcha
    @RequestMapping(path = "/campaign-forecast", method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<WebResponse> getCampaignForecast(
            @RequestBody CpmForecastRequest request,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin)
            throws JsonProcessingException {
        DirectAuthentication auth = authSource.getAuthentication();
        ClientId clientId = auth.getSubjectUser().getClientId();
        Long operatorUid = auth.getOperator().getUid();
        String requestId = UUID.randomUUID().toString().toUpperCase();

        ValidationResult<CpmForecastRequest, Defect> validation = campaignForecastValidationService
                .validateCampaignForecastRequest(request, operatorUid, clientId);
        if (validation.hasAnyErrors()) {
            return new ResponseEntity<>(validationResultConversionService.buildValidationResponse(validation),
                    HttpStatus.BAD_REQUEST);
        }

        CpmForecastResult result = campaignForecastService.forecast(requestId, request);

        // Возвращаем 500 если случилась непредвиденная или неизвестная ошибка
        if (!result.isSuccessful() && !result.getErrors().hasAnyErrors()) {
            return new ResponseEntity<>(
                    validationResultConversionService.buildValidationResponse(result.getErrors()),
                    HttpStatus.INTERNAL_SERVER_ERROR);
        }

        CpmForecastResponse response = new CpmForecastResponse(requestId, result.getResult(),
                validationResultConversionService.buildWebValidationResult(result.getErrors()));
        log.info("Response: {}", JsonUtils.toJson(response));
        return new ResponseEntity<>(response, HttpStatus.OK);
    }

    @PreAuthorizeRead
    @DisableAutoCaptcha
    @RequestMapping(path = "/general-forecast", method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<WebResponse> getGeneralCampaignForecast(
            @RequestBody GeneralCpmRecommendationRequest request,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin)
    {
        DirectAuthentication auth = authSource.getAuthentication();
        User subjectUser = auth.getSubjectUser();
        ClientId clientId = subjectUser.getClientId();
        Client client = clientService.getClient(clientId);
        CurrencyCode workCurrency = client.getWorkCurrency();
        User operator = auth.getOperator();
        Long operatorUid = operator.getUid();
        String requestId = UUID.randomUUID().toString().toUpperCase();

        ValidationResult<GeneralCpmRecommendationRequest, Defect> validation = campaignForecastValidationService
                .validateCampaignForecastRequest(request, operatorUid, clientId);
        if (validation.hasAnyErrors()) {
            return new ResponseEntity<>(validationResultConversionService.buildValidationResponse(validation),
                    HttpStatus.BAD_REQUEST);
        }

        GeneralCpmRecommendationResult result = inventoriWebService
                .forecast(request, workCurrency);

        // Возвращаем 500 если случилась непредвиденная или неизвестная ошибка
        if (!result.isSuccessful() && !result.getErrors().hasAnyErrors()) {
            return new ResponseEntity<>(
                    validationResultConversionService.buildValidationResponse(result.getErrors()),
                    HttpStatus.INTERNAL_SERVER_ERROR);
        }

        GeneralRecommendationResponse response = new GeneralRecommendationResponse(requestId, result.getResult(),
                validationResultConversionService.buildWebValidationResult(result.getErrors()));
        log.info("Response: {}", JsonUtils.toJson(response));
        return new ResponseEntity<>(response, HttpStatus.OK);
    }

    @ApiOperation(
            value = "getCpmTrafficLightPrediction",
            httpMethod = "POST",
            nickname = "getCpmTrafficLightPrediction"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 200, message = "Ok", response = CpmTrafficLightPredictionResponse.class),
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 501, message = "Not implemented", response = WebErrorResponse.class),
                    @ApiResponse(code = 504, message = "Gateway Timeout", response = WebErrorResponse.class),
            }
    )
    @PreAuthorizeRead
    @DisableAutoCaptcha
    @RequestMapping(path = "/cpm-traffic-light-prediction", method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<WebResponse> getCpmTrafficLightPrediction(
            @RequestBody CpmForecastRequest request,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin)
            throws JsonProcessingException {
        DirectAuthentication auth = authSource.getAuthentication();
        ClientId clientId = auth.getSubjectUser().getClientId();
        Long operatorUid = auth.getOperator().getUid();
        String requestId = UUID.randomUUID().toString().toUpperCase();

        ValidationResult<CpmForecastRequest, Defect> validation =
                campaignForecastValidationService
                        .validateCpmTrafficLightPredictionRequest(request, clientId, operatorUid);

        if (validation.hasAnyErrors()) {
            return new ResponseEntity<>(validationResultConversionService.buildValidationResponse(validation),
                    HttpStatus.BAD_REQUEST);
        }

        CpmTrafficLightPredictionResult result = campaignForecastService.trafficLightPrediction(requestId, request);

        // Возвращаем 500 если случилась непредвиденная или неизвестная ошибка
        if (!result.isSuccessful() && !result.getErrors().hasAnyErrors()) {
            return new ResponseEntity<>(
                    validationResultConversionService.buildValidationResponse(result.getErrors()),
                    HttpStatus.INTERNAL_SERVER_ERROR);
        }

        CpmTrafficLightPredictionResponse response = new CpmTrafficLightPredictionResponse(requestId,
                result.getResult(), validationResultConversionService.buildWebValidationResult(result.getErrors()));

        log.info("Response: {}", JsonUtils.toJson(response));
        return new ResponseEntity<>(response, HttpStatus.OK);
    }
}
