package ru.yandex.direct.web.entity.inventori.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.inventori.InventoriClient;
import ru.yandex.direct.inventori.model.request.CampaignPredictionRequest;
import ru.yandex.direct.inventori.model.request.Target;
import ru.yandex.direct.inventori.model.response.CampaignPredictionAvailableResponse;
import ru.yandex.direct.inventori.model.response.CampaignPredictionLowReachResponse;
import ru.yandex.direct.inventori.model.response.CampaignPredictionResponse;
import ru.yandex.direct.inventori.model.response.TrafficLightPredictionResponse;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.web.core.entity.inventori.model.CpmForecastRequest;
import ru.yandex.direct.web.core.entity.inventori.model.CpmForecastResult;
import ru.yandex.direct.web.core.entity.inventori.model.CpmForecastSuccessResult;
import ru.yandex.direct.web.core.entity.inventori.model.CpmTrafficLightPredictionResult;
import ru.yandex.direct.web.core.entity.inventori.model.CpmTrafficLightPredictionSuccessResult;
import ru.yandex.direct.web.core.entity.inventori.model.ForecastSector;
import ru.yandex.direct.web.core.entity.inventori.service.InventoriService;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Arrays.asList;
import static java.util.Collections.singleton;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.web.core.entity.inventori.service.InventoriService.lowReach;
import static ru.yandex.direct.web.core.entity.inventori.service.InventoriWebService.INVALID_GROUP_TYPES;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriDefectIds.Gen.CONTAINS_KEYWORD_ADGROUPS;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriDefectIds.Gen.NO_SUITABLE_ADGROUPS;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriDefectIds.Gen.UNSUPPORTED_ERROR;

/**
 * Сервис, считающий прогноз покрытия и выдающий рекомендацию по цене за показ для CPM-кампании.
 * <p>
 * Собирает все данные кампании по группам объявлений, и ходит в прогнозатор за прогнозом.
 */
@ParametersAreNonnullByDefault
@Component
public class CampaignForecastService {
    private static final long MIN_CPM_BID = 5_000_000L;

    private final DirectWebAuthenticationSource authProvider;
    private final ClientService clientService;
    private final UserService userService;
    private final InventoriClient inventoriClient;
    private final InventoriService inventoriService;

    @Autowired
    public CampaignForecastService(DirectWebAuthenticationSource authProvider,
                                   ClientService clientService,
                                   UserService userService,
                                   InventoriClient inventoriClient,
                                   InventoriService inventoriService) {
        this.authProvider = authProvider;
        this.clientService = clientService;
        this.userService = userService;
        this.inventoriClient = inventoriClient;
        this.inventoriService = inventoriService;
    }

    public CpmForecastSuccessResult forecastFake(ClientId clientId) {
        Client client = checkNotNull(clientService.getClient(clientId));
        CurrencyCode currency = client.getWorkCurrency().getCurrency().getCode();

        return defaultResult(currency);
    }

    public CpmForecastResult forecast(String requestId, CpmForecastRequest request) throws JsonProcessingException {
        AuthenticationInfo authInfo = getAuthInfo();

        CampaignPredictionRequest inventoriRequest = inventoriService
                .convertForecastRequest(authInfo.getOperatorUid(), authInfo.getClientId(), request,
                        authInfo.getCurrency());

        if (inventoriRequest.getTargets() != null && !inventoriRequest.getTargets().isEmpty()) {
            inventoriRequest.setTargets(filterInvalidTargets(inventoriRequest.getTargets()));
            if (inventoriRequest.getTargets().isEmpty()) {
                return new CpmForecastResult(request,
                        null,
                        singleton(new Defect<>(UNSUPPORTED_ERROR)));
            }
        }

        // проверяем, что есть необходимые данные - группы в нужных статусах
        if (inventoriRequest.getCampaignId() != null && inventoriRequest.getTargets().isEmpty()) {
            return new CpmForecastResult(request,
                    null,
                    singleton(noSuitableAdGroups()));
        }

        CampaignPredictionResponse response = inventoriClient.getCampaignPrediction(requestId,
                authInfo.getOperatorLogin(),
                authInfo.getClientLogin(),
                inventoriRequest);

        CpmForecastSuccessResult result = null;
        StreamEx<Defect> errors = StreamEx.of(response.getErrors()).map(inventoriService::convertError);
        if (inventoriRequest.containsKeywordGroup()) {
            errors = errors.append(containsKeywordGroup());
        }

        if (response instanceof CampaignPredictionLowReachResponse) {
            CampaignPredictionLowReachResponse typedResponse = (CampaignPredictionLowReachResponse) response;
            errors = errors.append(lowReach(typedResponse.getReachLessThan()));
        }

        if (response instanceof CampaignPredictionAvailableResponse) {
            CampaignPredictionAvailableResponse typedResponse = (CampaignPredictionAvailableResponse) response;
            if (typedResponse.getRecommendedCpm() != null || typedResponse.getRecommendedCpv() != null) {
                result = convertResponseToSuccessResult(typedResponse, authInfo.getCurrency());
            }
        }

        return new CpmForecastResult(request, result, errors.toList());
    }

    /**
     * Делает запрос в CPM-прогнозатор для получения цвета светофора и рекомендуемой цены и возвращает обработанный
     * результат.
     *
     * @param requestId id запроса
     * @param request   запрос
     * @return обработанный ответ от прогнозатора
     */
    public CpmTrafficLightPredictionResult trafficLightPrediction(String requestId,
                                                                  CpmForecastRequest request) throws JsonProcessingException {
        AuthenticationInfo authInfo = getAuthInfo();

        CampaignPredictionRequest inventoriRequest = inventoriService.convertTrafficLightPredictionRequest(
                authInfo.getOperatorUid(), authInfo.getClientId(), request, authInfo.getCurrency());

        if (inventoriRequest.getTargets() != null && !inventoriRequest.getTargets().isEmpty()) {
            inventoriRequest.setTargets(filterInvalidTargets(inventoriRequest.getTargets()));
            if (inventoriRequest.getTargets().isEmpty()) {
                return new CpmTrafficLightPredictionResult(request,
                        null,
                        singleton(new Defect<>(UNSUPPORTED_ERROR)));
            }
        }

        // проверяем, что есть необходимые данные - группы в нужных статусах
        if (inventoriRequest.getCampaignId() != null && inventoriRequest.getTargets().isEmpty()) {
            return new CpmTrafficLightPredictionResult(request,
                    null,
                    singleton(noSuitableAdGroups()));
        }

        CampaignPredictionResponse response = inventoriClient.getTrafficLightPrediction(requestId,
                authInfo.getOperatorLogin(),
                authInfo.getClientLogin(),
                inventoriRequest);

        CpmTrafficLightPredictionSuccessResult result = null;
        StreamEx<Defect> errors = StreamEx.of(response.getErrors()).map(inventoriService::convertError);
        if (inventoriRequest.containsKeywordGroup()) {
            errors = errors.append(containsKeywordGroup());
        }

        if (response instanceof CampaignPredictionLowReachResponse) {
            CampaignPredictionLowReachResponse typedResponse = (CampaignPredictionLowReachResponse) response;
            errors = errors.append(lowReach(typedResponse.getReachLessThan()));
        }

        if (response instanceof TrafficLightPredictionResponse) {
            TrafficLightPredictionResponse typedResponse = (TrafficLightPredictionResponse) response;
            result = convertResponseToSuccessResult(typedResponse, authInfo.getCurrency());
        }

        return new CpmTrafficLightPredictionResult(request, result, errors.toList());
    }

    private AuthenticationInfo getAuthInfo() {
        DirectAuthentication auth = authProvider.getAuthentication();
        User subClient = auth.getSubjectUser();
        ClientId clientId = subClient.getClientId();
        String clientLogin = subClient.getChiefUid() == null || subClient.getChiefUid().equals(subClient.getUid())
                ? subClient.getLogin()
                : userService.getChiefsLoginsByClientIds(singleton(clientId)).get(clientId);
        Client client = checkNotNull(clientService.getClient(clientId));
        CurrencyCode currency = client.getWorkCurrency().getCurrency().getCode();

        User operator = auth.getOperator();
        String operatorLogin = operator.getLogin();


        return new AuthenticationInfo(operatorLogin, clientLogin, currency, clientId, operator.getUid());
    }

    private CpmForecastSuccessResult defaultResult(CurrencyCode currency) {
        return new CpmForecastSuccessResult(
                inventoriService.convertToClientsCurrency(100_000_000, currency),
                asList(sector("red", 5_000_000, 15_000_000, currency),
                        sector("yellow", 45_000_000, 65_000_000, currency),
                        sector("green", 115_000_000, 300_000_000, currency)));
    }

    private CpmForecastSuccessResult convertResponseToSuccessResult(CampaignPredictionAvailableResponse response,
                                                                    CurrencyCode currency) {
        Long recommended = nvl(response.getRecommendedCpm(), response.getRecommendedCpv());
        Long redToYellowGradientStart = nvl(response.getCpmForRedToYellowGradientStart(), response.getCpvForRedToYellowGradientStart());
        Long redToYellowGradientEnd = nvl(response.getCpmForRedToYellowGradientEnd(), response.getCpvForRedToYellowGradientEnd());
        Long yellowToGreenGradientStart = nvl(response.getCpmForYellowToGreenGradientStart(), response.getCpvForYellowToGreenGradientStart());
        Long yellowToGreenGradientEnd = nvl(response.getCpmForYellowToGreenGradientEnd(), response.getCpvForYellowToGreenGradientEnd());

        checkNotNull(recommended);
        checkNotNull(redToYellowGradientStart);
        checkNotNull(redToYellowGradientEnd);
        checkNotNull(yellowToGreenGradientStart);
        checkNotNull(yellowToGreenGradientEnd);

        boolean isCpmStrategy = response.getRecommendedCpm() != null;
        BigDecimal minCurrencyBid = isCpmStrategy ? currency.getCurrency().getMinCpmPrice() : currency.getCurrency().getMinAvgCpv();
        long minimalBid = minCurrencyBid.multiply(BigDecimal.valueOf(1_000_000)).longValue();

        return new CpmForecastSuccessResult(
                inventoriService.convertToClientsCurrency(recommended, currency),
                asList(sector("red",
                        minimalBid, redToYellowGradientStart, currency),
                        sector("yellow",
                                redToYellowGradientEnd,
                                yellowToGreenGradientStart,
                                currency),
                        sector("green",
                                yellowToGreenGradientEnd,
                                currency)));
    }

    private CpmTrafficLightPredictionSuccessResult convertResponseToSuccessResult(
            TrafficLightPredictionResponse response,
            CurrencyCode currency) {
        checkNotNull(response.getTrafficLightColor());

        return new CpmTrafficLightPredictionSuccessResult(response.getTrafficLightColor(),
                response.getRecommendedCpm() == null ?
                        null : inventoriService.convertToClientsCurrency(response.getRecommendedCpm(), currency));
    }

    private ForecastSector sector(String color, long start, long end, CurrencyCode currency) {
        return new ForecastSector(color, inventoriService.convertToClientsCurrency(start, currency),
                inventoriService.convertToClientsCurrency(end, currency));
    }

    private ForecastSector sector(String color, long start, CurrencyCode currency) {
        return new ForecastSector(color, inventoriService.convertToClientsCurrency(start, currency));
    }

    private Defect noSuitableAdGroups() {
        return new Defect<>(NO_SUITABLE_ADGROUPS);
    }

    private Defect containsKeywordGroup() {
        return new Defect<>(CONTAINS_KEYWORD_ADGROUPS);
    }

    private static final class AuthenticationInfo {

        private final String operatorLogin;
        private final String clientLogin;
        private final CurrencyCode currency;
        private final ClientId clientId;
        private final Long operatorUid;

        private AuthenticationInfo(String operatorLogin, String clientLogin, CurrencyCode currency, ClientId clientId,
                                   Long operatorUid) {
            this.operatorLogin = operatorLogin;
            this.clientLogin = clientLogin;
            this.currency = currency;
            this.clientId = clientId;
            this.operatorUid = operatorUid;
        }

        public String getOperatorLogin() {
            return operatorLogin;
        }

        public String getClientLogin() {
            return clientLogin;
        }

        public CurrencyCode getCurrency() {
            return currency;
        }

        public ClientId getClientId() {
            return clientId;
        }

        public Long getOperatorUid() {
            return operatorUid;
        }
    }

    public List<Target> filterInvalidTargets(List<Target> targets) {
        return StreamEx.of(new ArrayList<>(targets))
                .nonNull()
                .filter(target -> !INVALID_GROUP_TYPES.contains(target.getGroupType()))
                .collect(Collectors.toList());
    }
}
