package ru.yandex.direct.web.entity.keyword.service;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.AdGroupNewMinusKeywords;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.UpdateAdGroupMinusKeywordsOperation;
import ru.yandex.direct.core.entity.campaign.container.CampaignNewMinusKeywords;
import ru.yandex.direct.core.entity.campaign.service.AppendCampaignMinusKeywordsOperation;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.WebOperationAggregatorWithNames;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.keyword.adqualityexport.AdQualityExportLogger;
import ru.yandex.direct.web.entity.keyword.model.AddAdGroupMinusKeywordsRequestItem;
import ru.yandex.direct.web.entity.keyword.model.AddAdGroupMinusKeywordsResultItem;
import ru.yandex.direct.web.entity.keyword.model.AddCampaignMinusKeywordsRequestItem;
import ru.yandex.direct.web.entity.keyword.model.AddCampaignMinusKeywordsResultItem;
import ru.yandex.direct.web.entity.keyword.model.AddMinusKeywordsRequest;
import ru.yandex.direct.web.entity.keyword.model.AddMinusKeywordsResponse;
import ru.yandex.direct.web.entity.keyword.model.AddMinusKeywordsResult;
import ru.yandex.direct.web.entity.keyword.presentation.AddAdGroupMinusKeywordsRequestPathConverters;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.web.validation.model.ValidationResponse;
import ru.yandex.direct.web.validation.model.WebValidationResult;

import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentState;
import static ru.yandex.direct.web.entity.keyword.model.AddMinusKeywordsRequest.AD_GROUP_MINUS_KEYWORDS;
import static ru.yandex.direct.web.entity.keyword.model.AddMinusKeywordsRequest.CAMPAIGN_MINUS_KEYWORDS;

@Service
public class AddMinusKeywordService {

    private final AdGroupService adGroupService;
    private final CampaignService campaignService;
    private final DirectWebAuthenticationSource authenticationSource;
    private final ValidationResultConversionService validationResultConversionService;
    private final PathNodeConverterProvider pathNodeConverterProvider;
    private final AdQualityExportLogger adQualityExportLogger;
    private final ClientGeoService clientGeoService;

    @Autowired
    public AddMinusKeywordService(AdGroupService adGroupService,
                                  CampaignService campaignService,
                                  DirectWebAuthenticationSource authenticationSource,
                                  PathNodeConverterProvider pathNodeConverterProvider,
                                  AdQualityExportLogger adQualityExportLogger,
                                  ValidationResultConversionService validationResultConversionService,
                                  ClientGeoService clientGeoService) {
        this.adGroupService = adGroupService;
        this.campaignService = campaignService;
        this.authenticationSource = authenticationSource;
        this.validationResultConversionService = validationResultConversionService;
        this.adQualityExportLogger = adQualityExportLogger;
        this.clientGeoService = clientGeoService;

        this.pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(AdGroup.class,
                        AddAdGroupMinusKeywordsRequestPathConverters.ADD_ADGROUP_MINUSKEYWORD_PATH_CONVERTER)
                .fallbackTo(pathNodeConverterProvider)
                .build();
    }

    public WebResponse addMinusKeywords(AddMinusKeywordsRequest request) {
        ValidationResult<AddMinusKeywordsRequest, Defect> requestValidationResult = validateRequest(request);
        if (hasValidationIssues(requestValidationResult)) {
            WebValidationResult requestWebValidationResult =
                    validationResultConversionService.buildWebValidationResult(requestValidationResult);
            return new ValidationResponse(requestWebValidationResult);
        }

        DirectAuthentication directAuthentication = authenticationSource.getAuthentication();
        long operatorUid = directAuthentication.getOperator().getUid();
        ClientId clientId = directAuthentication.getSubjectUser().getClientId();

        Optional<AppendCampaignMinusKeywordsOperation> optionalCampaignOperation =
                buildOptionalCampaignOperation(request, operatorUid, clientId);

        Optional<UpdateAdGroupMinusKeywordsOperation> optionalAdGroupOperation =
                buildOptionalAdGroupOperation(request, operatorUid, clientId);

        WebOperationAggregatorWithNames operationAggregator = new WebOperationAggregatorWithNames.Builder()
                .setValidationResultConverter((vr, path) -> validationResultConversionService
                        .buildWebValidationResult(vr, path, pathNodeConverterProvider))
                .addNullableOperation(CAMPAIGN_MINUS_KEYWORDS, optionalCampaignOperation.orElse(null))
                .addNullableOperation(AD_GROUP_MINUS_KEYWORDS, optionalAdGroupOperation.orElse(null))
                .build();

        Optional<WebValidationResult> optionalWebValidationResult = operationAggregator.executeAllOrNothing();
        if (optionalWebValidationResult.isPresent()) {
            return new ValidationResponse(optionalWebValidationResult.get());
        }
        if (optionalCampaignOperation.isPresent()) {
            AppendCampaignMinusKeywordsOperation operation = optionalCampaignOperation.get();
            adQualityExportLogger.logCampaignMinusKeywords(request.getCampaignMinusKeywords(),
                    operation.getOrigPhraseToNormPhrase());
        }

        if (optionalAdGroupOperation.isPresent()) {
            UpdateAdGroupMinusKeywordsOperation operation = optionalAdGroupOperation.get();
            adQualityExportLogger.logAdGroupMinusKeywords(request.getAdGroupMinusKeywords(),
                    operation.getOrigPhraseToNormPhrase());
        }

        return createSuccessfulResponse(request,
                optionalCampaignOperation.orElse(null), optionalAdGroupOperation.orElse(null));
    }

    private ValidationResult<AddMinusKeywordsRequest, Defect> validateRequest(
            AddMinusKeywordsRequest request) {
        ValidationResult<AddMinusKeywordsRequest, Defect> validationResult = new ValidationResult<>(request);

        if (request.getAdGroupMinusKeywords() == null && request.getCampaignMinusKeywords() == null) {
            validationResult.addError(inconsistentState());
        }

        return validationResult;
    }

    private Optional<UpdateAdGroupMinusKeywordsOperation> buildOptionalAdGroupOperation(
            AddMinusKeywordsRequest request, long operatorUid, ClientId clientId) {
        UpdateAdGroupMinusKeywordsOperation operation = null;

        if (request.getAdGroupMinusKeywords() != null) {
            List<AdGroupNewMinusKeywords> adGroupNewKeywordsList = mapList(request.getAdGroupMinusKeywords(),
                    addItem -> new AdGroupNewMinusKeywords(addItem.getId(), addItem.getMinusKeywords()));

            GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
            operation = adGroupService.createAppendMinusKeywordsOperation(Applicability.FULL, adGroupNewKeywordsList,
                    geoTree, operatorUid, clientId);
        }

        return Optional.ofNullable(operation);
    }

    private Optional<AppendCampaignMinusKeywordsOperation> buildOptionalCampaignOperation(
            AddMinusKeywordsRequest request, long operatorUid, ClientId clientId) {
        AppendCampaignMinusKeywordsOperation operation = null;

        if (request.getCampaignMinusKeywords() != null) {
            List<CampaignNewMinusKeywords> campaignNewKeywordsList = mapList(request.getCampaignMinusKeywords(),
                    addItem -> new CampaignNewMinusKeywords(addItem.getId(), addItem.getMinusKeywords()));

            operation = campaignService.createAppendMinusKeywordsOperation(campaignNewKeywordsList,
                    operatorUid, clientId);
        }

        return Optional.ofNullable(operation);
    }

    private AddMinusKeywordsResponse createSuccessfulResponse(AddMinusKeywordsRequest request,
                                                              @Nullable AppendCampaignMinusKeywordsOperation campaignOperation,
                                                              @Nullable UpdateAdGroupMinusKeywordsOperation adGroupOperation) {
        List<AddCampaignMinusKeywordsResultItem> campaignMinusKeywordsResults =
                createCampaignMinusKeywordsResults(request, campaignOperation);
        List<AddAdGroupMinusKeywordsResultItem> adGroupMinusKeywordsResults =
                createAdGroupMinusKeywordsResults(request, adGroupOperation);

        AddMinusKeywordsResult result =
                new AddMinusKeywordsResult(campaignMinusKeywordsResults, adGroupMinusKeywordsResults);
        return new AddMinusKeywordsResponse(result);
    }

    private List<AddCampaignMinusKeywordsResultItem> createCampaignMinusKeywordsResults(
            AddMinusKeywordsRequest request,
            @Nullable AppendCampaignMinusKeywordsOperation campaignOperation) {
        if (campaignOperation == null) {
            return null;
        }

        Map<Long, List<String>> normalMinusKeywords = campaignOperation.getNormalMinusKeywords();
        Map<Long, Integer> sumMinusKeywordsLength = campaignOperation.getSumMinusKeywordsLength();
        Map<Long, Integer> actualAddedMinusKeywordsCount = campaignOperation.getActualAddedMinusKeywordsCount();

        Function<AddCampaignMinusKeywordsRequestItem, AddCampaignMinusKeywordsResultItem> createResultItem =
                requestMinusKeywordsItem -> {
                    Long campaignId = requestMinusKeywordsItem.getId();
                    List<String> campNormalMinusKeywords = normalMinusKeywords.get(campaignId);
                    Integer campActualAddMinusKeywordsCount = actualAddedMinusKeywordsCount.get(campaignId);
                    Integer campSumMinusKeywordsLength = sumMinusKeywordsLength.get(campaignId);
                    return new AddCampaignMinusKeywordsResultItem(campaignId, campNormalMinusKeywords,
                            campActualAddMinusKeywordsCount, campSumMinusKeywordsLength);
                };

        return mapList(request.getCampaignMinusKeywords(), createResultItem);
    }

    private List<AddAdGroupMinusKeywordsResultItem> createAdGroupMinusKeywordsResults(
            AddMinusKeywordsRequest request,
            @Nullable UpdateAdGroupMinusKeywordsOperation adGroupOperation) {
        if (adGroupOperation == null) {
            return null;
        }

        Map<Long, List<String>> normalMinusKeywords = adGroupOperation.getNormalMinusKeywords();
        Map<Long, Integer> sumMinusKeywordsLength = adGroupOperation.getSumMinusKeywordsLength();
        Map<Long, Integer> actualAddedMinusKeywordsCount = adGroupOperation.getActualAddedMinusKeywordsCount();

        Function<AddAdGroupMinusKeywordsRequestItem, AddAdGroupMinusKeywordsResultItem> createResultItem =
                requestMinusKeywordsItem -> {
                    Long adGroupId = requestMinusKeywordsItem.getId();
                    List<String> adGroupNormalMinusKeywords = normalMinusKeywords.get(adGroupId);
                    Integer adGroupActualAddMinusKeywordsCount = actualAddedMinusKeywordsCount.get(adGroupId);
                    Integer adGroupSumMinusKeywordsLength = sumMinusKeywordsLength.get(adGroupId);
                    return new AddAdGroupMinusKeywordsResultItem(adGroupId, adGroupNormalMinusKeywords,
                            adGroupActualAddMinusKeywordsCount, adGroupSumMinusKeywordsLength);
                };

        return mapList(request.getAdGroupMinusKeywords(), createResultItem);
    }
}
