package ru.yandex.direct.web.entity.keyword.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.keyword.service.KeywordInclusionService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.keyword.model.CheckMinusKeywordsInclusionRequest;
import ru.yandex.direct.web.entity.keyword.model.CheckMinusKeywordsInclusionRequestItem;
import ru.yandex.direct.web.entity.keyword.service.validation.CheckKeywordInclusionValidationService;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.web.validation.model.ValidationResponse;

import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;
import static ru.yandex.direct.web.entity.keyword.model.CheckMinusKeywordsInclusionRequest.ADGROUP_MINUS_KEYWORDS_FIELD_NAME;
import static ru.yandex.direct.web.entity.keyword.model.CheckMinusKeywordsInclusionRequest.CAMPAIGN_MINUS_KEYWORDS_FIELD_NAME;
import static ru.yandex.direct.web.entity.keyword.service.validation.CheckMinusKeywordInclusionValidator.checkInclusionValidator;

@Service
@ParametersAreNonnullByDefault
public class CheckKeywordInclusionService {
    private static final Logger logger = LoggerFactory.getLogger(CheckKeywordInclusionService.class);

    private final DirectWebAuthenticationSource authenticationSource;
    private final CheckKeywordInclusionValidationService checkKeywordInclusionValidationService;
    private final KeywordInclusionService keywordInclusionService;
    private final ValidationResultConversionService conversionService;

    @Autowired
    public CheckKeywordInclusionService(DirectWebAuthenticationSource authenticationSource,
                                        CheckKeywordInclusionValidationService checkKeywordInclusionValidationService,
                                        KeywordInclusionService keywordInclusionService,
                                        ValidationResultConversionService conversionService) {
        this.authenticationSource = authenticationSource;
        this.checkKeywordInclusionValidationService = checkKeywordInclusionValidationService;
        this.keywordInclusionService = keywordInclusionService;
        this.conversionService = conversionService;
    }

    /**
     * Проверяет минус-фразы на предмет полного включения ключевых фраз.
     * <p>
     * Под полным включением понимается включение множества подходящих фразе поисковых запросов.
     */
    public WebResponse checkKeywordInclusion(CheckMinusKeywordsInclusionRequest request) {
        User subjectUser = authenticationSource.getAuthentication().getSubjectUser();
        ClientId clientId = subjectUser.getClientId();
        logger.trace("checkKeywordInclusion for clientId: {}", clientId);

        ValidationResult<CheckMinusKeywordsInclusionRequest, Defect> validationResult =
                new ValidationResult<>(request);

        List<CheckMinusKeywordsInclusionRequestItem> adGroupMinusKeywords = request.getAdGroupMinusKeywords();
        if (adGroupMinusKeywords != null) {
            ValidationResult<List<CheckMinusKeywordsInclusionRequestItem>, Defect> subValidationResult =
                    validationResult.getOrCreateSubValidationResult(
                            field(ADGROUP_MINUS_KEYWORDS_FIELD_NAME), adGroupMinusKeywords);
            ValidationResult<List<CheckMinusKeywordsInclusionRequestItem>, Defect> responseItemsForAdGroups =
                    getWrongMinusKeywordsItems(adGroupMinusKeywords,
                            keywordInclusionService::getMinusKeywordsIncludedInPlusKeywordsForAdGroup, clientId);
            subValidationResult.merge(responseItemsForAdGroups);
        }

        List<CheckMinusKeywordsInclusionRequestItem> campaignMinusKeywords = request.getCampaignMinusKeywords();
        if (campaignMinusKeywords != null) {
            ValidationResult<List<CheckMinusKeywordsInclusionRequestItem>, Defect> subValidationResult =
                    validationResult.getOrCreateSubValidationResult(
                            field(CAMPAIGN_MINUS_KEYWORDS_FIELD_NAME), campaignMinusKeywords);
            ValidationResult<List<CheckMinusKeywordsInclusionRequestItem>, Defect> responseItemsForCampaigns =
                    getWrongMinusKeywordsItems(campaignMinusKeywords,
                            keywordInclusionService::getMinusKeywordsIncludedInPlusKeywordsForCampaign, clientId);
            subValidationResult.merge(responseItemsForCampaigns);
        }

        if (validationResult.hasAnyErrors()) {
            // были ошибки валидации или негативные результаты проверки включения минус-фраз
            return conversionService.buildValidationResponse(validationResult);
        }

        return new WebSuccessResponse();
    }

    /**
     * Для переданного списка минус-фраз с привязкой к ID группы или кампании возвращает те минус-фразы,
     * которые полностью вычитают какую-либо ключевую фразу группы или кампании.
     *
     * @param minusKeywordsWithParentId набор {@link CheckMinusKeywordsInclusionRequestItem} &ndash; пар {@code (parentId, keyword)}
     * @param calcWrongMinusKeywords    функция поиска плохих минус-фраз
     * @param clientId                  {@code ClientID}
     * @return {@link ValidationResponse} содержащий ошибки валидации и результаты проверки переданного набора
     * {@link CheckMinusKeywordsInclusionRequestItem}
     */
    private ValidationResult<List<CheckMinusKeywordsInclusionRequestItem>, Defect> getWrongMinusKeywordsItems(
            @Nullable List<CheckMinusKeywordsInclusionRequestItem> minusKeywordsWithParentId,
            WrongMinusKeywordCalculator calcWrongMinusKeywords,
            ClientId clientId) {
        if (minusKeywordsWithParentId == null) {
            return new ValidationResult<>(null);
        }

        // Валидируем список переданных фраз
        ValidationResult<List<CheckMinusKeywordsInclusionRequestItem>, Defect> validationResult =
                checkKeywordInclusionValidationService.validateRequest(minusKeywordsWithParentId);
        List<CheckMinusKeywordsInclusionRequestItem> validItems = getValidItems(validationResult);

        // группируем по parentId, чтобы передать в core
        Map<Long, List<String>> minusKeywordsByParentId = StreamEx.of(validItems)
                .mapToEntry(CheckMinusKeywordsInclusionRequestItem::getId,
                        CheckMinusKeywordsInclusionRequestItem::getMinusKeywords)
                .toMap();

        Map<Long, ? extends Collection<String>> wrongMinusKeywordsByParentId =
                calcWrongMinusKeywords.apply(minusKeywordsByParentId, clientId);

        ValidationResult<List<CheckMinusKeywordsInclusionRequestItem>, Defect> checkResult =
                ListValidationBuilder.<CheckMinusKeywordsInclusionRequestItem, Defect>of(
                        minusKeywordsWithParentId)
                        .checkEachBy(checkInclusionValidator(wrongMinusKeywordsByParentId))
                        .getResult();

        validationResult.merge(checkResult);

        return validationResult;
    }

    private interface WrongMinusKeywordCalculator {
        Map<Long, ? extends Collection<String>> apply(Map<Long, List<String>> minusKeywordsByParentId,
                                                      ClientId clientId);
    }
}
