package ru.yandex.direct.web.entity.keyword.stat.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.advq.SearchKeywordResult;
import ru.yandex.direct.advq.SearchRequest;
import ru.yandex.direct.advq.exception.AdvqClientException;
import ru.yandex.direct.advq.search.AdvqRequestKeyword;
import ru.yandex.direct.advq.search.Statistics;
import ru.yandex.direct.advq.search.StatisticsPhraseItem;
import ru.yandex.direct.core.entity.showcondition.model.ShowStatRequest;
import ru.yandex.direct.core.entity.showcondition.service.ShowStatConverter;
import ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType;
import ru.yandex.direct.web.entity.keyword.stat.model.KeywordStatShowsBulkEntry;
import ru.yandex.direct.web.entity.keyword.stat.model.KeywordStatShowsBulkRequest;
import ru.yandex.direct.web.entity.keyword.stat.model.KeywordStatShowsBulkResponse;
import ru.yandex.direct.web.entity.keyword.stat.model.KeywordStatShowsItem;
import ru.yandex.direct.web.entity.keyword.stat.model.KeywordStatShowsRequest;
import ru.yandex.direct.web.entity.keyword.stat.model.KeywordStatShowsResponse;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;


/**
 * Методы конвертации для {@link KeywordStatShowsService}
 * Методы не статичны для удобства тестирования
 */
@Service
class KeywordStatShowsConverter {

    private static final Logger logger = LoggerFactory.getLogger(KeywordStatShowsConverter.class);

    private final ShowStatConverter showStatConverter;

    @Autowired
    public KeywordStatShowsConverter(ShowStatConverter showStatConverter) {
        this.showStatConverter = showStatConverter;
    }

    /**
     * Преобразовать запрос для одной фразы в запрос для нескольких фраз.
     */
    KeywordStatShowsBulkRequest convertToBulk(KeywordStatShowsRequest request) {
        return new KeywordStatShowsBulkRequest()
                .withAdGroupType(request.getAdGroupType())
                .withPhrases(singletonList(request.getPhrase()))
                .withCommonMinusPhrases(request.getCommonMinusPhrases())
                .withGeo(request.getGeo())
                .withCampaignId(request.getCampaignId())
                .withAdGroupId(request.getAdGroupId())
                .withLibraryMinusPhrasesIds(request.getLibraryMinusPhrasesIds())
                .withNeedSearchedWith(request.getNeedSearchedWith());
    }

    /**
     * Преобразовать ответ для нескольких фраз в запрос для одной фразы (первой).
     */
    KeywordStatShowsResponse convertFromBulk(KeywordStatShowsBulkResponse response) {
        KeywordStatShowsBulkEntry entry = response.getResult().get(0);
        return new KeywordStatShowsResponse()
                .withAdvqErrors(entry.getAdvqErrors())
                .withSearchedWith(entry.getSearchedWith())
                .withSearchedAlso(entry.getSearchedAlso());
    }

    /**
     * Преобразовать ответ от Advq по фразам {@code keywords}.
     */
    KeywordStatShowsBulkResponse convertToResponse(List<AdvqRequestKeyword> keywords,
                                                   Map<AdvqRequestKeyword, SearchKeywordResult> keywordToAdvqResult,
                                                   GdAdGroupType adGroupType, Predicate<String> phraseFilter) {
        List<KeywordStatShowsBulkEntry> entries = StreamEx.of(keywords)
                .map(k -> buildKeywordStatShowsBulkEntry(k, keywordToAdvqResult, adGroupType, phraseFilter))
                .toList();
        return new KeywordStatShowsBulkResponse(entries);
    }

    private KeywordStatShowsBulkEntry buildKeywordStatShowsBulkEntry(AdvqRequestKeyword keyword,
                                                                     Map<AdvqRequestKeyword, SearchKeywordResult> advqResults,
                                                                     GdAdGroupType adGroupType,
                                                                     Predicate<String> phraseFilter) {
        SearchKeywordResult searchKeywordResult = advqResults.get(keyword);
        if (searchKeywordResult == null || searchKeywordResult.isError()) {
            throw new AdvqClientException("search error");
        }
        Statistics stat = searchKeywordResult.getResult().getStat();
        List<String> errors = stat.getErrors();
        if (CollectionUtils.isNotEmpty(errors)) {
            logger.error("Got errors for phrase {}, from ADVQ response: {}", keyword, errors);
        }
        return new KeywordStatShowsBulkEntry()
                .withShows(countShows(stat.getTotalCount(), adGroupType))
                .withAdvqErrors(errors)
                .withSearchedWith(toKeywordStatShowsItemsList(stat.getIncludingPhrases(), adGroupType, phraseFilter))
                .withSearchedAlso(toKeywordStatShowsItemsList(stat.getAssociations(), adGroupType, phraseFilter));
    }

    /**
     * Костыль для отдачи статистики по Коллекциям - в среднем колдунщик коллекций показывается 1 на 10 показов серпа,
     * поэтому для коллекций берём стандартный прогноз и делим на 10 с округлением вверх. См. DIRECT-106559
     * @param shows       Количество показов на серпе.
     * @param adGroupType Тип группы, для которой вычисляли статистику.
     * @return Если статистика вычислялась для Коллекций, возвращает показы, поделённые на 10,
     * иначе возвращает исходные показы.
     * @see ru.yandex.direct.grid.processing.service.showcondition.converter.RefineWordConverter#fixStatisticsForContentPromotionCollections(Statistics, GdAdGroupType)
     */
    private static long countShows(long shows, GdAdGroupType adGroupType) {
        if (adGroupType == GdAdGroupType.CONTENT_PROMOTION_COLLECTION) {
            return (shows + 9) / 10;
        } else {
            return shows;
        }
    }

    static List<KeywordStatShowsItem> toKeywordStatShowsItemsList(
            @Nullable Collection<StatisticsPhraseItem> phraseItems,
            GdAdGroupType adGroupType, Predicate<String> phraseFilter) {
        if (phraseItems == null || phraseItems.isEmpty()) {
            return emptyList();
        }
        return StreamEx.of(phraseItems)
                .filter(pi -> phraseFilter.test(pi.getPhrase()))
                .map(phraseItem -> toKeywordStatShowsItem(phraseItem, adGroupType))
                .toList();
    }

    private static KeywordStatShowsItem toKeywordStatShowsItem(
            StatisticsPhraseItem phraseItem,
            GdAdGroupType adGroupType) {
        return new KeywordStatShowsItem()
                .withPhrase(phraseItem.getPhrase())
                .withShows(countShows(phraseItem.getCnt(), adGroupType));
    }

    public ShowStatRequest convertRequest(KeywordStatShowsBulkRequest bulkRequest) {
        return new ShowStatRequest()
                .withGeo(bulkRequest.getGeo())
                .withCommonMinusPhrases(bulkRequest.getCommonMinusPhrases())
                .withPhrases(bulkRequest.getPhrases())
                .withCampaignId(bulkRequest.getCampaignId())
                .withAdGroupId(bulkRequest.getAdGroupId())
                .withLibraryMinusPhrasesIds(bulkRequest.getLibraryMinusPhrasesIds())
                .withDeviceTypes(bulkRequest.getDeviceTypes())
                .withGender(bulkRequest.getGender())
                .withAges(bulkRequest.getAges())
                .withIsContentPromotionVideo(bulkRequest.getAdGroupType() == GdAdGroupType.CONTENT_PROMOTION_VIDEO)
                .withNeedSearchedWith(bulkRequest.getNeedSearchedWith())
                .withNeedSearchedAlso(bulkRequest.getNeedSearchedAlso())
                .withLimit(bulkRequest.getLimit());
    }

    public SearchRequest convertRequest(ShowStatRequest request) {
        return showStatConverter.convertRequest(request);
    }
}
