package ru.yandex.direct.web.entity.keyword.stat.service;

import java.util.Map;
import java.util.function.Predicate;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.advq.SearchKeywordResult;
import ru.yandex.direct.advq.SearchRequest;
import ru.yandex.direct.advq.exception.AdvqClientException;
import ru.yandex.direct.advq.search.AdvqRequestKeyword;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseValidator;
import ru.yandex.direct.core.entity.showcondition.model.ShowStatRequest;
import ru.yandex.direct.core.entity.showcondition.service.ShowStatService;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.entity.keyword.stat.model.KeywordStatShowsBulkRequest;
import ru.yandex.direct.web.entity.keyword.stat.model.KeywordStatShowsBulkResponse;
import ru.yandex.direct.web.entity.keyword.stat.model.KeywordStatShowsRequest;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static ru.yandex.direct.libs.keywordutils.parser.KeywordParser.parseWithMinuses;

@Service
public class KeywordStatShowsService {

    private final ValidationResultConversionService validationResultConversionService;
    private final KeywordStatShowsValidationService keywordStatShowsValidationService;
    private final KeywordStatShowsConverter keywordStatShowsConverter;
    private final StopWordService stopWordService;
    private final KeywordWithLemmasFactory keywordFactory;
    private final ShowStatService showStatService;

    @Autowired
    public KeywordStatShowsService(ValidationResultConversionService validationResultConversionService,
                                   KeywordStatShowsValidationService keywordStatShowsValidationService,
                                   KeywordStatShowsConverter keywordStatShowsConverter,
                                   StopWordService stopWordService,
                                   KeywordWithLemmasFactory keywordFactory,
                                   ShowStatService showStatService) {
        this.validationResultConversionService = validationResultConversionService;
        this.keywordStatShowsValidationService = keywordStatShowsValidationService;
        this.keywordStatShowsConverter = keywordStatShowsConverter;
        this.stopWordService = stopWordService;
        this.keywordFactory = keywordFactory;
        this.showStatService = showStatService;
    }

    public WebResponse getKeywordStatShows(KeywordStatShowsRequest request) {
        KeywordStatShowsBulkRequest bulkRequest = keywordStatShowsConverter.convertToBulk(request);
        ValidationResult<KeywordStatShowsBulkRequest, Defect> rv =
                keywordStatShowsValidationService.validate(bulkRequest);
        if (rv.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(rv);
        }
        try {
            KeywordStatShowsBulkResponse bulkResponse = getKeywordStatShowBulkResponse(bulkRequest);
            return keywordStatShowsConverter.convertFromBulk(bulkResponse);
        } catch (AdvqClientException e) {
            return new WebErrorResponse(HttpStatus.INTERNAL_SERVER_ERROR.value(), e.getMessage());
        }
    }

    public WebResponse getKeywordStatShowsBulk(KeywordStatShowsBulkRequest bulkRequest) {
        ValidationResult<KeywordStatShowsBulkRequest, Defect> rv =
                keywordStatShowsValidationService.validate(bulkRequest);
        if (rv.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(rv);
        }
        try {
            return getKeywordStatShowBulkResponse(bulkRequest);
        } catch (AdvqClientException e) {
            return new WebErrorResponse(HttpStatus.INTERNAL_SERVER_ERROR.value(), e.getMessage());
        }
    }

    private KeywordStatShowsBulkResponse getKeywordStatShowBulkResponse(KeywordStatShowsBulkRequest bulkRequest) {
        ShowStatRequest coreRequest = keywordStatShowsConverter.convertRequest(bulkRequest);
        SearchRequest advqRequest = keywordStatShowsConverter.convertRequest(coreRequest);
        Map<AdvqRequestKeyword, SearchKeywordResult> advqResultByKeyWord = showStatService.getStatShowByPhrase(coreRequest,
                advqRequest);
        return keywordStatShowsConverter.convertToResponse(advqRequest.getKeywords(), advqResultByKeyWord,
                bulkRequest.getAdGroupType(), getPhraseFilter());
    }

    private Predicate<String> getPhraseFilter() {
        return phrase -> {
            if (phrase == null) {
                return true;
            } else {
                var validationResult = new PhraseValidator(stopWordService, keywordFactory, parseWithMinuses(phrase))
                        .apply(phrase);
                return !validationResult.hasAnyErrors();
            }
        };
    }
}
