package ru.yandex.direct.web.entity.log;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.common.log.container.LogType;
import ru.yandex.direct.common.log.service.CommonDataLogService;
import ru.yandex.direct.web.annotations.AllowedOperatorRoles;
import ru.yandex.direct.web.annotations.AllowedSubjectRoles;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.core.security.authentication.exception.DirectAuthenticationServiceException;
import ru.yandex.direct.web.core.security.captcha.DisableAutoCaptcha;
import ru.yandex.direct.web.core.security.csrf.CsrfCheck;
import ru.yandex.direct.web.entity.log.model.LogDataRequest;
import ru.yandex.direct.web.entity.log.model.TimingsRequest;

import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.CLIENT;
import static ru.yandex.direct.rbac.RbacRole.EMPTY;
import static ru.yandex.direct.rbac.RbacRole.MANAGER;
import static ru.yandex.direct.rbac.RbacRole.MEDIA;
import static ru.yandex.direct.rbac.RbacRole.PLACER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;

@RestController
@RequestMapping(value = "/log",
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "log")
public class LogController {
    private final CommonDataLogService commonDataLogService;
    private final SolomonLogService solomonLogService;
    private final DirectWebAuthenticationSource auth;

    @Autowired
    public LogController(CommonDataLogService commonDataLogService,
                         SolomonLogService solomonLogService,
                         DirectWebAuthenticationSource auth) {
        this.commonDataLogService = commonDataLogService;
        this.solomonLogService = solomonLogService;
        this.auth = auth;
    }

    @ApiOperation(
            value = "frontendTimings",
            httpMethod = "POST",
            nickname = "frontendTimings",
            notes = "Log information about page rendering timings to common_data.log"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "Ok", response = WebSuccessResponse.class)
    })
    @RequestMapping(path = "/frontendTimings",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    @CsrfCheck(enabled = false)
    @DisableAutoCaptcha
    @AllowedOperatorRoles({SUPER, SUPERREADER, SUPPORT, PLACER, MEDIA, MANAGER, AGENCY, CLIENT, EMPTY})
    @AllowedSubjectRoles({SUPER, SUPERREADER, SUPPORT, PLACER, MEDIA, MANAGER, AGENCY, CLIENT, EMPTY})
    public WebResponse frontendTimings(@RequestBody TimingsRequest request) {
        commonDataLogService.log(LogType.FRONTEND_TIMINGS, request);
        solomonLogService.record(request.getTimings());
        return new WebSuccessResponse();
    }

    @ApiOperation(
            value = "dna",
            httpMethod = "POST",
            nickname = "dna",
            notes = "Log custom information to common_data.log"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "Ok", response = WebSuccessResponse.class)
    })
    @RequestMapping(path = "/dna",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    @CsrfCheck(enabled = false)
    @DisableAutoCaptcha
    @AllowedOperatorRoles({SUPER, SUPERREADER, SUPPORT, PLACER, MEDIA, MANAGER, AGENCY, CLIENT, EMPTY})
    @AllowedSubjectRoles({SUPER, SUPERREADER, SUPPORT, PLACER, MEDIA, MANAGER, AGENCY, CLIENT, EMPTY})
    public WebResponse dna(@RequestBody LogDataRequest request) {
        try {
            request.setOperatorUid(auth.getAuthentication().getOperator().getUid());
        } catch (DirectAuthenticationServiceException e) {
            //Если нет аутентификации. Может потом захотим как-то еще помечать
            request.setOperatorUid(0);
        }
        commonDataLogService.log(LogType.DNA, request);
        return new WebSuccessResponse();
    }

}
