package ru.yandex.direct.web.entity.log;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.solomon.SolomonUtils.FRONTEND_TIMINGS_METRICS_REGISTRY;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Сервисный класс записи frontend timing в Solomon
 */
@Service
public class SolomonLogService {
    private static final Logger LOGGER = LoggerFactory.getLogger(SolomonLogService.class);
    private static final int MAX_SENSOR_COUNT = 200;
    private Map<String, Histogram> sensors = new ConcurrentHashMap<>();
    private Map<String, Histogram> frontendTimingsMetricsSensors = new ConcurrentHashMap<>();

    public void record(Map<String, Double> timings) {
        for (Map.Entry<String, Double> entry : timings.entrySet()) {
            if (entry.getValue() == null) {
                continue;
            }
            if (sensors.containsKey(entry.getKey())) {
                Histogram sensor = sensors.get(entry.getKey());
                sensor.record(entry.getValue().longValue());

                Histogram frontendTimingsMetricsSensor = frontendTimingsMetricsSensors.get(entry.getKey());
                frontendTimingsMetricsSensor.record(entry.getValue().longValue());
            } else {
                recordSensorAndSaveToSensors(entry, FRONTEND_TIMINGS_METRICS_REGISTRY, frontendTimingsMetricsSensors);
                if (sensors.size() < MAX_SENSOR_COUNT) {
                    recordSensorAndSaveToSensors(entry, SOLOMON_REGISTRY, sensors);
                } else {
                    LOGGER.warn("Maximum number of Solomon sensors exceeded. Sensor {} skipped", entry.getKey());
                }
            }
        }
    }

    private void recordSensorAndSaveToSensors(Map.Entry<String, Double> entry, MetricRegistry metricsRegistry,
                                              Map<String, Histogram> sensors) {
        Histogram sensor = metricsRegistry.histogramRate(
                "frontend_timing",
                Labels.of("func", entry.getKey()),
                Histograms.exponential(12, 2, 100)
        );
        sensor.record(entry.getValue().longValue());
        sensors.put(entry.getKey(), sensor);
    }
}
