package ru.yandex.direct.web.entity.misprints.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiResponse
import io.swagger.annotations.ApiResponses
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.MediaType
import org.springframework.stereotype.Controller
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestMethod
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.ResponseBody
import ru.yandex.direct.core.entity.banner.type.body.BannerWithBodyConstants
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.web.core.model.WebErrorResponse
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider
import ru.yandex.direct.web.entity.MassResponse
import ru.yandex.direct.web.entity.banner.model.BannerImagesUploadResponse
import ru.yandex.direct.web.entity.misprints.service.CheckMisprintService
import ru.yandex.direct.web.validation.model.WebDefect
import ru.yandex.direct.web.validation.model.WebValidationResult

class WebCheckMissprintsRespone : MassResponse<List<CheckMisprintResponse>>()

data class CheckMisprintResponse(
        val code: Int,
        val pos: Int,
        val row: Int,
        val col: Int,
        val len: Int,
        val word: String,
        val s: List<String>
)

@Controller
@RequestMapping(value = ["/textvalidation"], produces = [MediaType.APPLICATION_JSON_UTF8_VALUE])
@Api(tags = ["text_validation"])
open class CheckMisprintController @Autowired constructor(
        private val checkMisprintService: CheckMisprintService
) {

    @ApiOperation(value = "checkMisprints", httpMethod = "GET", nickname = "checkMisprints")
    @ApiResponses(
            ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse::class),
            ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse::class),
            ApiResponse(code = 200, message = "Ok", response = BannerImagesUploadResponse::class)
    )
    @PreAuthorizeRead
    @RequestMapping(path = ["checkMisprints"], method = [RequestMethod.GET])
    @ResponseBody
    open fun checkMisprints(@RequestParam(value = "text", required = true) textToCheck: String,
                        @SuppressWarnings("unused")
                        @RequestParam(value = DirectCookieAuthProvider.PARAMETER_ULOGIN) subjectLogin: String): WebCheckMissprintsRespone {

        val validationResult = validate(textToCheck)
        if (validationResult.validationResult()?.errors?.isNotEmpty() == true) {
            return validationResult
        }
        val result = checkMisprintService.checkMisprints(textToCheck)
        return WebCheckMissprintsRespone().withResult(
                result.map { entry ->
                    CheckMisprintResponse(
                            code = 1,
                            pos = entry.key,
                            row = 0,
                            col = entry.key,
                            len = entry.value.length,
                            word = entry.value,
                            s = listOf()
                    )
                }.toList()
        )
    }

    fun validate(textToCheck: String): WebCheckMissprintsRespone {
        if (textToCheck.isBlank()) {
            return WebCheckMissprintsRespone()
                    .withValidation(WebValidationResult().addErrors(
                            WebDefect().withText("Text to check is empty")))
        }
        if (textToCheck.length > maxOf(BannerWithBodyConstants.MAX_LENGTH_BODY,
                        BannerWithBodyConstants.MAX_LENGTH_MOBILE_BODY,
                        BannerWithBodyConstants.MAX_LENGTH_CONTENT_PROMOTION_BODY)) {
            return WebCheckMissprintsRespone()
                    .withValidation(WebValidationResult().addErrors(
                            WebDefect().withText("Text to check is too long")))
        }
        return WebCheckMissprintsRespone();
    }

}
