package ru.yandex.direct.web.entity.mobileapptracker.model;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;

import ru.yandex.direct.web.core.entity.mobilecontent.model.TrackingSystem;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebOsType;
import ru.yandex.direct.web.entity.mobileapptracker.TrackerUrlUtils;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.singleton;

@ParametersAreNonnullByDefault
public class WebMobileAppTrackerUrlTemplate {
    private final TrackingSystem trackingSystem;

    private final String trackingSystemName;

    // шаблон, куда должен подставиться trackerId и, обычно, в нём должен быть параметр {logid}
    private final String baseUrlTemplate;

    // тут может быть пустой список
    private final List<WebMobileAppTrackerUrlParameter> parameters;

    @SuppressWarnings("WeakerAccess")
    public WebMobileAppTrackerUrlTemplate(TrackingSystem trackingSystem, String trackingSystemName,
                                          String baseUrlTemplate, List<WebMobileAppTrackerUrlParameter> parameters) {
        this.trackingSystem = trackingSystem;
        this.trackingSystemName = trackingSystemName;
        this.baseUrlTemplate = baseUrlTemplate;
        this.parameters = parameters;
    }

    @JsonProperty(required = true)
    public TrackingSystem getTrackingSystem() {
        return trackingSystem;
    }

    @JsonProperty(required = true)
    public String getTrackingSystemName() {
        return trackingSystemName;
    }

    @JsonProperty(required = true)
    public String getBaseUrlTemplate() {
        return baseUrlTemplate;
    }

    @JsonProperty(required = true)
    public String getUrlTemplateIos() {
        return getUrlTemplate(singleton(WebOsType.IOS));
    }

    @JsonProperty(required = true)
    public String getUrlTemplateAndroid() {
        return getUrlTemplate(singleton(WebOsType.ANDROID));
    }

    // Метод нужен для обратной совместимости. Выпилить без сожаления, когда релизнится DIRECT-81888 в dna
    @JsonProperty(required = true)
    public String getUrlTemplate() {
        return getUrlTemplate(ImmutableSet.of(WebOsType.IOS, WebOsType.ANDROID));
    }

    public String getUrlTemplate(Set<WebOsType> osTypes) {
        return StreamEx.of(getParameters())
                .filter(p -> p.getOsType() == null || osTypes.contains(p.getOsType()))
                .map(p -> TrackerUrlUtils.createParamValuePairString(p.getName(), p.getValue()))
                .prepend(baseUrlTemplate)
                .collect(TrackerUrlUtils.joinUrlCollector());
    }

    @JsonProperty(required = true)
    public List<WebMobileAppTrackerUrlParameter> getParameters() {
        return parameters;
    }

    @JsonProperty(required = true)
    public boolean isEditable() {
        return baseUrlTemplate.isEmpty();
    }

    public static Builder builder() {
        return new Builder();
    }

    @SuppressWarnings("unused")
    @ParametersAreNonnullByDefault
    public static final class Builder {
        private TrackingSystem trackingSystem;
        private String trackingSystemName;
        private String baseUrlTemplate;
        private List<WebMobileAppTrackerUrlParameter> parameters;

        private Builder() {
        }

        public static Builder aTrackerUrlTemplateInfo2() {
            return new Builder();
        }

        public Builder withTrackingSystem(TrackingSystem trackingSystem) {
            this.trackingSystem = trackingSystem;
            return this;
        }

        public Builder withTrackingSystemName(String trackingSystemName) {
            this.trackingSystemName = trackingSystemName;
            return this;
        }

        public Builder withBaseUrlTemplate(String baseUrlTemplate) {
            this.baseUrlTemplate = baseUrlTemplate;
            return this;
        }

        public Builder withParameters(List<WebMobileAppTrackerUrlParameter> parameters) {
            this.parameters = ImmutableList.copyOf(parameters);
            return this;
        }

        public WebMobileAppTrackerUrlTemplate build() {
            return new WebMobileAppTrackerUrlTemplate(
                    checkNotNull(trackingSystem), checkNotNull(trackingSystemName), checkNotNull(baseUrlTemplate),
                    checkNotNull(parameters));
        }
    }
}
