package ru.yandex.direct.web.entity.mobilecontent.controller;

import java.net.URI;
import java.util.Optional;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.web.annotations.AllowedSubjectRoles;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebMobileApp;
import ru.yandex.direct.web.core.entity.mobilecontent.service.WebCoreMobileAppService;
import ru.yandex.direct.web.core.model.WebCaptchaResponse;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.mobilecontent.model.MobileAppCreateResponse;
import ru.yandex.direct.web.entity.mobilecontent.model.MobileAppGenerateIdResponse;
import ru.yandex.direct.web.entity.mobilecontent.model.MobileAppGetResponse;
import ru.yandex.direct.web.entity.mobilecontent.model.MobileAppListResponse;
import ru.yandex.direct.web.entity.mobilecontent.model.MobileAppUpdateResponse;
import ru.yandex.direct.web.entity.mobilecontent.model.MobileContentInfoResponse;
import ru.yandex.direct.web.entity.mobilecontent.model.WebUpdateMobileAppRequest;
import ru.yandex.direct.web.entity.mobilecontent.service.WebUpdateMobileAppService;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.rbac.RbacRole.CLIENT;
import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;

@Controller
@AllowedSubjectRoles({CLIENT})
@RequestMapping(value = "/mobile_app",
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "mobile_app")
@SuppressWarnings("unused")
public class MobileAppController {
    private final DirectWebAuthenticationSource authenticationSource;
    private final WebCoreMobileAppService webCoreMobileAppService;
    private final WebUpdateMobileAppService webUpdateMobileAppService;
    private final ValidationResultConversionService validationResultConversionService;
    private final PathNodeConverterProvider pathNodeConverterProvider;

    public MobileAppController(
            DirectWebAuthenticationSource authenticationSource,
            WebCoreMobileAppService webCoreMobileAppService,
            WebUpdateMobileAppService webUpdateMobileAppService,
            ValidationResultConversionService validationResultConversionService,
            PathNodeConverterProvider pathNodeConverterProvider) {
        this.authenticationSource = authenticationSource;
        this.webCoreMobileAppService = webCoreMobileAppService;
        this.webUpdateMobileAppService = webUpdateMobileAppService;
        this.validationResultConversionService = validationResultConversionService;
        this.pathNodeConverterProvider = pathNodeConverterProvider;
    }

    @PreAuthorizeWrite
    @ApiOperation(
            value = "GenerateId",
            nickname = "GenerateId",
            httpMethod = "POST"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 429, message = "Too Many Requests", response = WebCaptchaResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = MobileAppCreateResponse.class)
            }
    )
    @RequestMapping(path = "/generate_id", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse generateId() {
        Long id = webCoreMobileAppService.generateIds(1).get(0);
        return new MobileAppGenerateIdResponse().withResult(id);
    }

    @PreAuthorizeWrite
    @ApiOperation(
            value = "createMobileApp",
            nickname = "createMobileApp",
            httpMethod = "POST"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 429, message = "Too Many Requests", response = WebCaptchaResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = MobileAppCreateResponse.class)
            }
    )
    @RequestMapping(path = "/create", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse create(
            @RequestBody WebMobileApp webMobileApp,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        DirectAuthentication auth = authenticationSource.getAuthentication();
        User operator = auth.getOperator();
        User subjectUser = auth.getSubjectUser();
        ClientId clientId = subjectUser.getClientId();
        MassResult<Long> result = webCoreMobileAppService.addApps(
                operator, clientId, singletonList(webMobileApp));
        if (hasValidationIssues(result)) {
            return validationResultConversionService.buildValidationResponse(
                    result.getValidationResult(), pathNodeConverterProvider);
        } else {
            Long createdMobileAppId = result.get(0).getResult();
            return webCoreMobileAppService.getApp(clientId, createdMobileAppId)
                    .map(MobileAppCreateResponse::new)
                    .orElseThrow(() -> new IllegalStateException("Couldn't retrieve just created mobile app"));
        }
    }


    @PreAuthorizeWrite
    @ApiOperation(
            value = "updateMobileApp",
            nickname = "updateMobileApp",
            httpMethod = "POST"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 404, message = "Not found", response = WebErrorResponse.class),
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 429, message = "Too Many Requests", response = WebCaptchaResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = MobileAppUpdateResponse.class)
            }
    )
    @RequestMapping(path = "/update", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse updateMobileApp(
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin,
            @RequestBody WebUpdateMobileAppRequest updateRequest
    ) {
        DirectAuthentication auth = authenticationSource.getAuthentication();
        User operator = auth.getOperator();
        User subjectUser = auth.getSubjectUser();
        ClientId clientId = subjectUser.getClientId();

        Result<Long> result = webUpdateMobileAppService.updateApp(operator, clientId, updateRequest);

        if (hasValidationIssues(result)) {
            return validationResultConversionService.buildValidationResponse(
                    result.getValidationResult(), pathNodeConverterProvider);
        } else {
            Long createdMobileAppId = result.getResult();
            return webCoreMobileAppService.getApp(clientId, createdMobileAppId)
                    .map(MobileAppUpdateResponse::new)
                    .orElseThrow(() -> new IllegalStateException("Couldn't retrieve updated mobile app"));
        }
    }

    @PreAuthorizeRead
    @ApiOperation(
            value = "getMobileApp",
            nickname = "getMobileApp",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 404, message = "Not found", response = WebErrorResponse.class),
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 429, message = "Too Many Requests", response = WebCaptchaResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = MobileAppGetResponse.class)
            }
    )
    @RequestMapping(path = "/get", method = RequestMethod.GET)
    @ResponseBody
    public WebResponse getMobileApp(
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin,
            @RequestParam(value = "id") long mobileAppId
    ) {
        DirectAuthentication auth = authenticationSource.getAuthentication();
        User subjectUser = auth.getSubjectUser();
        ClientId clientId = subjectUser.getClientId();
        return webCoreMobileAppService.getApp(clientId, mobileAppId)
                .map(MobileAppGetResponse::new)
                .map(WebResponse.class::cast)
                .orElse(new WebErrorResponse(404, "Mobile app not found"));
    }

    @PreAuthorizeRead
    @ApiOperation(
            value = "getMobileAppList",
            nickname = "getMobileAppList",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 429, message = "Too Many Requests", response = WebCaptchaResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = MobileAppListResponse.class)
            }
    )
    @RequestMapping(path = "/list", method = RequestMethod.GET)
    @ResponseBody
    public MobileAppListResponse getMobileAppList(
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        DirectAuthentication auth = authenticationSource.getAuthentication();
        User subjectUser = auth.getSubjectUser();
        ClientId clientId = subjectUser.getClientId();
        return new MobileAppListResponse(webCoreMobileAppService.getAppList(clientId, null, null));
    }

    @PreAuthorizeRead
    @ApiOperation(
            value = "getOldMobileAppList",
            nickname = "getOldMobileAppList",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 429, message = "Too Many Requests", response = WebCaptchaResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = MobileAppListResponse.class)
            }
    )
    @RequestMapping(path = "/old_list", method = RequestMethod.GET)
    @ResponseBody
    public MobileAppListResponse getOldMobileAppList(
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        DirectAuthentication auth = authenticationSource.getAuthentication();
        User subjectUser = auth.getSubjectUser();
        ClientId clientId = subjectUser.getClientId();
        return new MobileAppListResponse(webCoreMobileAppService.getOldAppList(clientId));
    }

    @PreAuthorizeRead
    @ApiOperation(
            value = "getMobileContent",
            nickname = "getMobileContent",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 429, message = "Too Many Requests", response = WebCaptchaResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = MobileContentInfoResponse.class)
            }
    )
    @RequestMapping(path = "/mobile_content", method = RequestMethod.GET)
    @ResponseBody
    public WebResponse getMobileContent(
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin,
            @RequestParam(value = "url") String url) {
        DirectAuthentication auth = authenticationSource.getAuthentication();
        User subjectUser = auth.getSubjectUser();
        ClientId clientId = subjectUser.getClientId();
        Optional<MobileAppStoreUrl> parsedUrl = MobileAppStoreUrlParser.parse(url);
        Optional<String> maybeStoreUrl = MobileAppStoreUrlParser.sanitizeUrl(url).map(URI::toString);
        if (!parsedUrl.isPresent() || !maybeStoreUrl.isPresent()) {
            return new WebErrorResponse(400, "Unable to parse URL: " + url);
        }
        return webCoreMobileAppService.getMobileContent(clientId, maybeStoreUrl.get(), parsedUrl.get(), true)
                .map(r -> (WebResponse) new MobileContentInfoResponse(r))
                .orElse(new WebErrorResponse(400, "Content not found: " + url));
    }
}
