package ru.yandex.direct.web.entity.mobilecontent.converter;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.web.core.entity.mobilecontent.converter.TrackerConverter;
import ru.yandex.direct.web.core.entity.mobilecontent.converter.WebCoreMobileAppConverter;
import ru.yandex.direct.web.entity.mobilecontent.model.WebUpdateMobileAppRequest;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.core.entity.mobilecontent.converter.WebCoreMobileAppConverter.DEVICE_TYPE_TARGETING_SET_TO_CORE_CONVERTER;
import static ru.yandex.direct.web.core.entity.mobilecontent.converter.WebCoreMobileAppConverter.DISPLAYED_ATTRIBUTE_SET_TO_CORE_CONVERTER;
import static ru.yandex.direct.web.core.entity.mobilecontent.converter.WebCoreMobileAppConverter.DOMAIN_CONVERTER;
import static ru.yandex.direct.web.core.entity.mobilecontent.converter.WebCoreMobileAppConverter.NETWORK_TARGETING_SET_TO_CORE_CONVERTER;

@Component
@ParametersAreNonnullByDefault
public class WebMobileAppConverter {
    private final TrackerConverter trackerConverter;

    public WebMobileAppConverter(
            TrackerConverter trackerConverter) {
        this.trackerConverter = trackerConverter;
    }

    /**
     * Сконвертировать веб-запрос на изменение данных мобильного приложения в изменение модели MobileApp
     *
     * @param updateRequest веб-запрос на изменение данных мобильного приложения
     * @return изменение модели MobileApp
     */
    public ModelChanges<MobileApp> createMobileAppModelChanges(WebUpdateMobileAppRequest updateRequest) {
        ModelChanges<MobileApp> mc = new ModelChanges<>(updateRequest.id, MobileApp.class);
        mc.processNotNull(updateRequest.name, MobileApp.NAME, String::trim);
        mc.process(updateRequest.displayedAttributes, MobileApp.DISPLAYED_ATTRIBUTES,
                DISPLAYED_ATTRIBUTE_SET_TO_CORE_CONVERTER::convert);
        mc.process(updateRequest.minOsVersion, MobileApp.MINIMAL_OPERATING_SYSTEM_VERSION);
        mc.process(updateRequest.deviceTypeTargeting, MobileApp.DEVICE_TYPE_TARGETING,
                DEVICE_TYPE_TARGETING_SET_TO_CORE_CONVERTER::convert);
        mc.process(updateRequest.networkTargeting, MobileApp.NETWORK_TARGETING,
                NETWORK_TARGETING_SET_TO_CORE_CONVERTER::convert);
        mc.processNotNull(updateRequest.domain, MobileApp.DOMAIN, DOMAIN_CONVERTER::convert);
        mc.process(trackerConverter.convertTrackersToCore(updateRequest.trackers), MobileApp.TRACKERS);
        mc.process(updateRequest.appMetrikaApplicationId, MobileApp.APP_METRIKA_APPLICATION_ID);

        mc.process(updateRequest.appMetrikaApplicationId == null
                        ? emptyList()
                        : mapList(updateRequest.mobileEvents, e -> WebCoreMobileAppConverter.
                        toCoreAppmetrikaEvent(e, updateRequest.appMetrikaApplicationId)),
                MobileApp.MOBILE_APP_METRIKA_EVENTS);

        mc.process(updateRequest.appMetrikaApplicationId == null
                        ? mapList(updateRequest.mobileEvents, e -> WebCoreMobileAppConverter
                        .toCoreExternalTrackerEvent(e, updateRequest.id))
                        : emptyList(),
                MobileApp.MOBILE_EXTERNAL_TRACKER_EVENTS);
        return mc;
    }
}
