package ru.yandex.direct.web.entity.mobilecontent.service;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.mobileapp.MobileAppDefects;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppUpdateOperation;
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppUpdateOperationFactory;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebMobileAppTracker;
import ru.yandex.direct.web.entity.mobilecontent.converter.WebMobileAppConverter;
import ru.yandex.direct.web.entity.mobilecontent.model.PropagationRequest;
import ru.yandex.direct.web.entity.mobilecontent.model.WebUpdateMobileAppRequest;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.listSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.web.core.entity.mobilecontent.service.WebCoreMobileAppService.validateEvent;
import static ru.yandex.direct.web.entity.mobilecontent.model.PropagationMode.DO_NOT_APPLY_TO_BANNERS;

@Service
public class WebUpdateMobileAppService {
    private final WebMobileAppConverter webMobileAppConverter;
    private final MobileAppUpdateOperationFactory mobileAppUpdateOperationFactory;
    private final MobileAppUpdatePropagationOperationFactory mobileAppUpdatePropagationOperationFactory;

    public WebUpdateMobileAppService(
            WebMobileAppConverter webMobileAppConverter,
            MobileAppUpdateOperationFactory mobileAppUpdateOperationFactory,
            MobileAppUpdatePropagationOperationFactory mobileAppUpdatePropagationOperationFactory) {
        this.webMobileAppConverter = webMobileAppConverter;
        this.mobileAppUpdateOperationFactory = mobileAppUpdateOperationFactory;
        this.mobileAppUpdatePropagationOperationFactory = mobileAppUpdatePropagationOperationFactory;
    }

    private ValidationResult<WebUpdateMobileAppRequest, Defect> validateUpdateRequest(
            @Nullable WebUpdateMobileAppRequest updateRequest) {
        if (updateRequest == null) {
            return ValidationResult.failed(null, CommonDefects.notNull());
        }
        ItemValidationBuilder<WebUpdateMobileAppRequest, Defect> vb =
                ItemValidationBuilder.of(updateRequest, Defect.class);
        vb.item(updateRequest.id, "id")
                .check(notNull(), When.isValid());
        vb.list(updateRequest.trackers, "trackers")
                .check(notNull(), When.isValid())
                .check(listSize(0, 1), When.isValid())
                .checkEachBy(this::validateWebTracker);
        vb.list(updateRequest.mobileEvents, "mobileEvents")
                .checkEachBy(event -> validateEvent(event, updateRequest.appMetrikaApplicationId != null),
                        When.notNull());
        return vb.getResult();
    }

    private ValidationResult<WebMobileAppTracker, Defect> validateWebTracker(WebMobileAppTracker tracker) {
        ItemValidationBuilder<WebMobileAppTracker, Defect> vb =
                ItemValidationBuilder.of(tracker, Defect.class);
        vb.item(tracker.getUrl(), WebMobileAppTracker.URL)
                .check(notNull())
                .check(notBlank());
        return vb.getResult();
    }

    public Result<Long> updateApp(User operator, ClientId clientId, WebUpdateMobileAppRequest updateRequest) {
        ValidationResult<WebUpdateMobileAppRequest, Defect> requestValidationResult =
                validateUpdateRequest(updateRequest);
        if (requestValidationResult.hasAnyErrors()) {
            return Result.broken(requestValidationResult);
        }

        PropagationRequest propagationRequest = PropagationRequest.builder()
                .withPropagationMode(nvl(updateRequest.propagationMode, DO_NOT_APPLY_TO_BANNERS))
                .withMobileAppId(updateRequest.id)
                .withDisplayedAttributes(nvl(updateRequest.displayedAttributes, Collections.emptySet()))
                .withTrackingSystem(updateRequest.trackers.isEmpty() ? null : updateRequest.trackers.get(0).getTrackingSystem().name())
                .withTrackingUrl(updateRequest.trackers.isEmpty() ? null : updateRequest.trackers.get(0).getUrl())
                .withTrackingImpressionUrl(updateRequest.trackers.isEmpty() ? null : updateRequest.trackers.get(0).getImpressionUrl())
                .build();

        MobileAppUpdatePropagationOperationFactory.Result maybeOperation =
                mobileAppUpdatePropagationOperationFactory.createPropagationOperation(
                        operator.getUid(), clientId, propagationRequest);
        if (!maybeOperation.hasOperation()) {
            return Result.broken(ValidationResult.failed(updateRequest, maybeOperation.getError()));
        }
        Operation<?> propagateOperation = maybeOperation.getOperation();

        MobileAppUpdateOperation updateMobileAppOperation =
                createMobileAppUpdateOperation(operator, clientId, updateRequest);
        Optional<MassResult<Long>> resultOfAppUpdatePreparing = updateMobileAppOperation.prepare();
        if (resultOfAppUpdatePreparing.isPresent()) {
            // операция обновления приложения не прошла валидацию
            return resultOfAppUpdatePreparing.get().get(0);
        }
        Optional<? extends MassResult<?>> resultOfPropagationPreparing = propagateOperation.prepare();
        if (resultOfPropagationPreparing.isPresent()) {
            // операция пропагации не прошла валидацию
            return Result.broken(ValidationResult.failed(updateRequest, MobileAppDefects.propagationIsFailed()));
        }

        Result<Long> resultOfAppUpdate = updateMobileAppOperation.apply().get(0);
        propagateOperation.apply();

        return resultOfAppUpdate;
    }

    private MobileAppUpdateOperation createMobileAppUpdateOperation(User operator, ClientId clientId,
                                                                    WebUpdateMobileAppRequest updateRequest) {
        List<ModelChanges<MobileApp>> changes = Collections.singletonList(
                webMobileAppConverter.createMobileAppModelChanges(updateRequest));
        return mobileAppUpdateOperationFactory.createFullOperation(operator, clientId, changes);
    }
}
