package ru.yandex.direct.web.entity.moderation.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.core.security.SecurityTranslations;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.web.annotations.AllowedOperatorRoles;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.moderation.model.RemoderateBannerPageRequest;
import ru.yandex.direct.web.entity.moderation.service.RemoderateBannerPageService;

import static ru.yandex.direct.rbac.RbacRole.MANAGER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;


@RestController
@RequestMapping(path = "bannerPagesModeration",
        consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "bannerPagesModeration")
public class BannerPagesModerationController {

    private final RemoderateBannerPageService remoderateBannerPageService;
    private final DirectWebAuthenticationSource authenticationSource;
    private final RbacService rbacService;

    @Autowired
    public BannerPagesModerationController(
            RemoderateBannerPageService remoderateBannerPageService,
            DirectWebAuthenticationSource authenticationSource, RbacService rbacService) {
        this.remoderateBannerPageService = remoderateBannerPageService;
        this.authenticationSource = authenticationSource;
        this.rbacService = rbacService;
    }

    @ApiOperation(value = "remoderateBannerPage", httpMethod = "POST", nickname = "remoderateBannerPage")
    @ApiResponses(
            {
                    @ApiResponse(code = 200, message = "Ok", response = WebResponse.class)
            }
    )
    @AllowedOperatorRoles({RbacRole.SUPER, RbacRole.SUPERREADER, RbacRole.SUPPORT, RbacRole.MANAGER})
    @PostMapping(path = "/remoderate_banner_page")
    public WebResponse remoderateBannerPage(
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN) String subjectLogin,
            @RequestBody RemoderateBannerPageRequest input
    ) {
        checkBannerPagesRemoderationRights();
        return remoderateBannerPageService.remoderate(input);
    }

    /**
     * Ручку могут вызвать только super, support, manager и SupperReader-developer.
     * Из-за SupperReader-developer нельзя использовать просто @PreAuthorizeWrite + @AllowedOperatorRoles.
     */
    private void checkBannerPagesRemoderationRights() {
        User operator = authenticationSource.getAuthentication().getOperator();
        User client = authenticationSource.getAuthentication().getSubjectUser();

        // бросаем exception, если это обычный суперридер или менеджер который редактирует чужого клиента.
        if ((MANAGER.equals(operator.getRole()) && !rbacService.isOwner(operator.getUid(), client.getUid()))
                || (SUPERREADER.equals(operator.getRole()) && !operator.getDeveloper())
        ) {
            throw new AccessDeniedException(
                    String.format("У оператора [uid=%s, role=%s, isDeveloper=false] нет прав на выполнение операции",
                            operator.getUid(),
                            operator.getRole()),
                    SecurityTranslations.INSTANCE.accessDenied());
        }
    }
}
