package ru.yandex.direct.web.entity.moderation.service;

import java.util.List;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.ModerateBannerPagesRepository;
import ru.yandex.direct.core.entity.moderation.repository.ModerationRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.moderation.model.RemoderateBannerPageRequest;
import ru.yandex.direct.web.entity.moderation.service.validation.RemoderateBannerPageValidationService;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static java.util.Collections.singleton;

/**
 * Сброс вердикта модерации баннера внешними операторами (на текущий момент только outdoor баннеры/операторы)
 * (таблица moderate_banner_pages)
 */
@Service
public class RemoderateBannerPageService {

    private final DirectWebAuthenticationSource authenticationSource;
    private final RemoderateBannerPageValidationService validationService;
    private final ValidationResultConversionService conversionService;
    private final ModerateBannerPagesRepository bannerPagesRepository;
    private final ModerationRepository moderationRepository;
    private final BannerCommonRepository bannerRepository;
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public RemoderateBannerPageService(
            DirectWebAuthenticationSource authenticationSource,
            RemoderateBannerPageValidationService validationService,
            ValidationResultConversionService conversionService,
            ModerateBannerPagesRepository bannerPagesRepository,
            ModerationRepository moderationRepository,
            BannerCommonRepository bannerRepository,
            DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.authenticationSource = authenticationSource;
        this.validationService = validationService;
        this.conversionService = conversionService;
        this.bannerPagesRepository = bannerPagesRepository;
        this.moderationRepository = moderationRepository;
        this.bannerRepository = bannerRepository;
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    public WebResponse remoderate(RemoderateBannerPageRequest input) {
        ClientId clientId = authenticationSource.getAuthentication().getSubjectUser().getClientId();

        ValidationResult<RemoderateBannerPageRequest, Defect> vr = validationService.validate(input, clientId);
        if (vr.hasAnyErrors()) {
            return conversionService.buildValidationResponse(vr);
        }

        resetStatusAndDeleteReason(clientId, input.getBannerId(), input.getPageIds());

        return new WebSuccessResponse();
    }

    /**
     * Сбросить статус модерации на ready в таблице moderate_banner_pages, удалить данные о модерации из mod_reasons
     * и сбросить флаг statusBsSynced у баннера.
     *
     * @param clientId - клиент
     * @param bannerId - идентификатор баннера
     * @param pageIds  - идентификаторы операторов щитов
     */
    private void resetStatusAndDeleteReason(ClientId clientId, Long bannerId, List<Long> pageIds) {
        int shard = shardHelper.getShardByClientId(clientId);

        dslContextProvider.ppcTransaction(shard, conf -> {
            Set<Long> moderateBannerPageIds = bannerPagesRepository.getModerateBannerPageIds(shard, bannerId, pageIds);
            bannerPagesRepository.resetModerateStatus(conf.dsl(), moderateBannerPageIds);
            moderationRepository.deleteBannerPageModReason(conf.dsl(), moderateBannerPageIds);
            bannerRepository.resetStatusBsSyncedByIds(conf.dsl(), singleton(bannerId));
        });
    }
}
