package ru.yandex.direct.web.entity.moderation.service.validation;

import java.util.List;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.repository.ModerateBannerPagesRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.moderation.model.RemoderateBannerPageRequest;

import static java.util.Collections.singleton;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.web.entity.moderation.model.RemoderateBannerPageRequest.Prop.BANNER_ID_FIELD_NAME;
import static ru.yandex.direct.web.entity.moderation.model.RemoderateBannerPageRequest.Prop.PAGE_IDS_FIELD_NAME;

@Service
public class RemoderateBannerPageValidationService {

    private final ModerateBannerPagesRepository bannerPagesRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public RemoderateBannerPageValidationService(
            ModerateBannerPagesRepository bannerPagesRepository,
            BannerTypedRepository bannerTypedRepository,
            ShardHelper shardHelper) {
        this.bannerPagesRepository = bannerPagesRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.shardHelper = shardHelper;
    }

    /**
     * Проверить что bannerId принадлежит клиенту и существует модерация для наборов pageIds + bannerId
     *
     * @param input    - входные данные
     * @param clientId - клиент
     * @return - результат валидации
     */
    public ValidationResult<RemoderateBannerPageRequest, Defect> validate(
            RemoderateBannerPageRequest input, ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);

        ItemValidationBuilder<RemoderateBannerPageRequest, Defect> ivb = ItemValidationBuilder.of(input);

        ivb.item(input.getBannerId(), BANNER_ID_FIELD_NAME)
                .check(notNull())
                .check(validId())
                .checkBy(bannerId -> checkBannerBelongsToClient(shard, clientId, bannerId), When.isValid());

        if (ivb.getResult().hasAnyErrors()) {
            return ivb.getResult();
        }

        ivb.list(input.getPageIds(), PAGE_IDS_FIELD_NAME)
                .check(notNull())
                .checkEach(notNull())
                .checkEach(validId())
                .checkBy(pageIds -> checkBannerPageIdsExisting(shard, pageIds, input.getBannerId()), When.isValid());

        return ivb.getResult();
    }

    /**
     * Проверить, что баннер принадлежит клиенту
     */
    private ValidationResult<Long, Defect> checkBannerBelongsToClient(int shard, ClientId clientId, Long bannerId) {
        Set<Long> simpleBanner = bannerTypedRepository.getClientBannerIds(shard, clientId, singleton(bannerId));

        return ItemValidationBuilder.of(bannerId, Defect.class)
                .check(fromPredicate(x -> !simpleBanner.isEmpty(), CommonDefects.objectNotFound()))
                .getResult();
    }

    /**
     * Проверить, что в таблице moderate_banner_pages есть все наборы pageIds + bannerId.
     */
    private ValidationResult<List<Long>, Defect> checkBannerPageIdsExisting(int shard, List<Long> pageIds,
                                                                            Long bannerId) {
        Set<Long> existingModeratePageIds = bannerPagesRepository.getExistingModeratePageIds(shard, bannerId, pageIds);

        return ListValidationBuilder.of(pageIds, Defect.class)
                .checkEach(fromPredicate(existingModeratePageIds::contains, CommonDefects.objectNotFound()))
                .getResult();
    }
}
