package ru.yandex.direct.web.entity.retargetinglists.service;

import java.util.List;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.web.core.model.retargeting.MetrikaGoalWeb;
import ru.yandex.direct.web.core.model.retargeting.RetargetingConditionConverter;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;

import static ru.yandex.direct.core.entity.retargeting.model.GoalType.AUDIENCE;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.CDP_SEGMENT;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.ECOMMERCE;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.GOAL;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.LAL_SEGMENT;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.MOBILE;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.SEGMENT;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;


@Service
public class GetRetargetingGoalsService {
    private static final Logger logger = LoggerFactory.getLogger(GetRetargetingGoalsService.class);

    //Добавление типа в список должно быть поддержано со стороны фронта
    private static final Set<GoalType> GOAL_TYPES_FOR_RETARGETING = ImmutableSet.of(
            GOAL,
            SEGMENT,
            ECOMMERCE,
            AUDIENCE,
            LAL_SEGMENT,
            CDP_SEGMENT,
            MOBILE
    );

    private final DirectWebAuthenticationSource authenticationSource;
    private final RetargetingConditionService retargetingConditionService;


    @Autowired
    public GetRetargetingGoalsService(
            DirectWebAuthenticationSource authenticationSource,
            RetargetingConditionService retargetingConditionService) {
        this.authenticationSource = authenticationSource;
        this.retargetingConditionService = retargetingConditionService;
    }

    /**
     * Получение списка целей
     *
     * @return цели для интерфейса
     */
    public List<MetrikaGoalWeb> getMetrikaGoalsForRetargeting() {
        return getMetrikaGoalsForRetargetingByGoalTypes(GOAL_TYPES_FOR_RETARGETING);
    }

    /**
     * Получение списка целей
     *
     * @param goalTypes типы целей для фильтра
     * @return цели для интерфейса
     */
    public List<MetrikaGoalWeb> getMetrikaGoalsForRetargetingByGoalTypes(Set<GoalType> goalTypes) {
        User subClient = authenticationSource.getAuthentication().getSubjectUser();
        logger.debug("uid: {}, clientId: {}", subClient.getUid(), subClient.getClientId());
        List<Goal> goals = retargetingConditionService.getMetrikaGoalsForRetargeting(subClient.getClientId());
        return mapList(
                filterList(goals, goal -> goalTypes.contains(goal.getType())),
                goal -> (MetrikaGoalWeb) RetargetingConditionConverter.fromGoal(goal)
        );
    }
}
