package ru.yandex.direct.web.entity.retargetinglists.service;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.retargeting.container.SwitchRetargeting;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.retargetinglists.model.SwitchRetargetingConverter;
import ru.yandex.direct.web.entity.retargetinglists.model.SwitchRetargetingWeb;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;


@Service
public class SwitchRetargetingConditionsService {
    private static final Logger logger = LoggerFactory.getLogger(SwitchRetargetingConditionsService.class);

    private final DirectWebAuthenticationSource authenticationSource;
    private final RetargetingService retargetingService;
    private final ValidationResultConversionService validationResultConversionService;
    private final SwitchRetargetingConditionValidationService switchRetargetingConditionValidationService;


    @Autowired
    public SwitchRetargetingConditionsService(
            DirectWebAuthenticationSource authenticationSource,
            RetargetingService retargetingService,
            ValidationResultConversionService validationResultConversionService,
            SwitchRetargetingConditionValidationService switchRetargetingConditionValidationService) {
        this.authenticationSource = authenticationSource;
        this.retargetingService = retargetingService;
        this.validationResultConversionService = validationResultConversionService;
        this.switchRetargetingConditionValidationService = switchRetargetingConditionValidationService;
    }

    public WebResponse switchRetargetingConditions(List<SwitchRetargetingWeb> switchRetargetingWebs) {
        User subclient = authenticationSource.getAuthentication().getSubjectUser();
        logger.debug("uid: {}, clientId: {}", subclient.getUid(), subclient.getClientId().asLong());

        ValidationResult<List<SwitchRetargetingWeb>, Defect> preValidationResult =
                switchRetargetingConditionValidationService.validate(switchRetargetingWebs);

        if (preValidationResult.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(preValidationResult);
        }

        MassResult<SwitchRetargeting> operationResult =
                retargetingService.switchRetargetingConditions(
                        mapList(switchRetargetingWebs, SwitchRetargetingConverter::toSwitchRetargeting),
                        subclient.getClientId(),
                        Applicability.FULL);
        if (hasValidationIssues(operationResult)) {
            return validationResultConversionService.buildValidationResponse(operationResult);
        }
        return new WebSuccessResponse();
    }

    public WebResponse switchRetargetingConditions(List<Long> retCondIds, Boolean status) {
        return switchRetargetingConditions(mapList(retCondIds,
                retCondId -> new SwitchRetargetingWeb().withRetCondId(retCondId).withSuspended(status)));
    }
}
