package ru.yandex.direct.web.entity.retargetinglists.service;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionOperationFactory;
import ru.yandex.direct.core.entity.retargeting.service.UpdateRetargetingConditionsConfig;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.retargeting.RetargetingConditionActionResponse;
import ru.yandex.direct.web.core.model.retargeting.RetargetingConditionWeb;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.web.core.model.retargeting.RetargetingConditionConverter.toRetargetingCondition;

@Service
public class UpdateRetargetingConditionsService {
    private static final Logger logger = LoggerFactory.getLogger(UpdateRetargetingConditionsService.class);

    private final RetargetingConditionOperationFactory retargetingConditionOperationFactory;
    private final DirectWebAuthenticationSource authenticationSource;
    private final RetargetingConditionWebValidationService retargetingConditionValidator;
    private final ValidationResultConversionService validationResultConversionService;

    @Autowired
    public UpdateRetargetingConditionsService(RetargetingConditionOperationFactory retargetingConditionOperationFactory,
                                              DirectWebAuthenticationSource authenticationSource,
                                              RetargetingConditionWebValidationService retargetingConditionValidator,
                                              ValidationResultConversionService validationResultConversionService) {
        this.retargetingConditionOperationFactory = retargetingConditionOperationFactory;
        this.authenticationSource = authenticationSource;
        this.retargetingConditionValidator = retargetingConditionValidator;
        this.validationResultConversionService = validationResultConversionService;
    }

    public WebResponse updateCondition(RetargetingConditionWeb retargetingConditionWeb) {
        DirectAuthentication authentication = authenticationSource.getAuthentication();
        ClientId clientId = authentication.getSubjectUser().getClientId();
        long uid = authentication.getSubjectUser().getUid();
        logger.debug("uid: {}, clientId: {}", uid, clientId);

        return updateCondition(retargetingConditionWeb, clientId);
    }

    public ValidationResult<?, Defect> updateUacCondition(
            RetargetingConditionWeb retargetingConditionWeb, ClientId clientId) {
        return updateConditionWithValidationResult(retargetingConditionWeb, clientId);
    }

    private ValidationResult<?, Defect> updateConditionWithValidationResult(
            RetargetingConditionWeb retargetingConditionWeb, ClientId clientId) {
        ValidationResult<RetargetingConditionWeb, Defect> preValidationResult =
                retargetingConditionValidator.validate(retargetingConditionWeb);

        if (preValidationResult.hasAnyErrors()) {
            return preValidationResult;
        }

        RetargetingCondition coreRetargetingCondition =
                toRetargetingCondition(
                        retargetingConditionWeb, clientId);

        ModelChanges<RetargetingCondition> modelChanges =
                toRetargetingConditionModelChanges(coreRetargetingCondition);
        MassResult<Long> operationResult = retargetingConditionOperationFactory
                .updateRetargetingConditions(clientId, singletonList(modelChanges),
                        new UpdateRetargetingConditionsConfig(Applicability.FULL));

        if (hasValidationIssues(operationResult)) {
            Result singleResult = operationResult.getResult()
                    .stream()
                    .findFirst()
                    .orElse((Result) operationResult);
            return singleResult.getValidationResult();
        }

        return ValidationResult.success(coreRetargetingCondition.getId());
    }

    private WebResponse updateCondition(RetargetingConditionWeb retargetingConditionWeb, ClientId clientId) {
        ValidationResult<?, Defect> updateResult =
                updateConditionWithValidationResult(retargetingConditionWeb, clientId);
        if (updateResult.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(updateResult);
        } else {
            return new RetargetingConditionActionResponse((Long) updateResult.getValue());
        }
    }

    /**
     * Формирует {@link ModelChanges} для {@link RetargetingCondition}. Web шлет name/description/rules вне зависимости от того,
     * поменялись они или нет, поэтому результат включает в себя {@link RetargetingCondition#NAME},
     * {@link RetargetingCondition#DESCRIPTION} и {@link RetargetingCondition#RULES}(если непустая коллекция)
     */
    private ModelChanges<RetargetingCondition> toRetargetingConditionModelChanges(
            RetargetingCondition coreRetargetingCondition) {
        ModelChanges<RetargetingCondition> modelChanges =
                new ModelChanges<>(coreRetargetingCondition.getId(), RetargetingCondition.class);
        modelChanges.processNotNull(coreRetargetingCondition.getName(), RetargetingCondition.NAME)
                .processNotNull(coreRetargetingCondition.getDescription(), RetargetingCondition.DESCRIPTION);
        if (coreRetargetingCondition.getRules() != null) {
            modelChanges.processNotNull(coreRetargetingCondition.getRules(), RetargetingCondition.RULES);
        }
        return modelChanges;
    }
}
