package ru.yandex.direct.web.entity.security.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiResponse
import io.swagger.annotations.ApiResponses
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.HttpStatus.OK
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.stereotype.Controller
import org.springframework.util.LinkedMultiValueMap
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.ResponseBody
import ru.yandex.direct.rbac.RbacRole.AGENCY
import ru.yandex.direct.rbac.RbacRole.CLIENT
import ru.yandex.direct.rbac.RbacRole.EMPTY
import ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_ADMIN
import ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_MANAGER
import ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_SUPERREADER
import ru.yandex.direct.rbac.RbacRole.LIMITED_SUPPORT
import ru.yandex.direct.rbac.RbacRole.MANAGER
import ru.yandex.direct.rbac.RbacRole.MEDIA
import ru.yandex.direct.rbac.RbacRole.PLACER
import ru.yandex.direct.rbac.RbacRole.SUPER
import ru.yandex.direct.rbac.RbacRole.SUPERREADER
import ru.yandex.direct.rbac.RbacRole.SUPPORT
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.web.annotations.AllowedSubjectRoles
import ru.yandex.direct.web.core.model.WebErrorResponse
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource
import ru.yandex.direct.web.core.security.captcha.DisableAutoCaptcha
import ru.yandex.direct.web.entity.security.model.CsrfTokenResponse
import ru.yandex.direct.web.entity.security.service.WebSecurityService
import javax.annotation.ParametersAreNonnullByDefault

@Controller
@ParametersAreNonnullByDefault
@RequestMapping(value = ["/security"], produces = [MediaType.APPLICATION_JSON_UTF8_VALUE])
@Api(tags = ["security"])
open class SecurityController @Autowired constructor(
        private val directWebAuthenticationSource: DirectWebAuthenticationSource,
        private val webSecurityService: WebSecurityService
) {

    @ApiOperation(value = "getCsrfToken", httpMethod = "GET", nickname = "getCsrfToken")
    @ApiResponses(
            ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse::class),
            ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse::class),
            ApiResponse(code = 200, message = "Ok", response = CsrfTokenResponse::class))
    @DisableAutoCaptcha
    @GetMapping(path = ["/get_csrf_token"])
    @AllowedSubjectRoles(SUPER, SUPERREADER, SUPPORT, PLACER, MEDIA, MANAGER, INTERNAL_AD_ADMIN, INTERNAL_AD_MANAGER,
            INTERNAL_AD_SUPERREADER, AGENCY, CLIENT, LIMITED_SUPPORT, EMPTY)
    @ResponseBody
    open fun getCsrfToken(): ResponseEntity<String> {
        val auth = directWebAuthenticationSource.authentication
        val csrfToken = webSecurityService.getCsrfToken(auth)
        val noCacheHeaders = noCacheHeaders()
        val body = JsonUtils.toJson(csrfToken)
        return ResponseEntity(body, noCacheHeaders, OK)
    }

    /**
     * Возвращает заголовки рекомендованные СИБ для предотвращения кеширования запроса:
     * https://wiki.yandex-team.ru/security/for/web-developers/csrf/#kakperedavattokenvvjorstku
     * Т.к. они во времени и контексте не изменяются, то нет особого смысла менять их на честные константны из
     * какого-либо фреймворка или библиотеки.
     */
    private fun noCacheHeaders(): LinkedMultiValueMap<String, String> {
        val headers = LinkedMultiValueMap<String, String>()
        headers.add("Cache-Control", "max-age=0, must-revalidate, proxy-revalidate, no-cache, no-store, private")
        headers.add("Expires", "Thu, 01 Jan 1970 00:00:01 GMT")
        headers.add("Pragma", "no-cache")
        headers.add("X-Content-Type-Options", "nosniff")
        headers.add("X-Frame-Options", "SAMEORIGIN")
        headers.add("x-dns-prefetch-control", "off")
        return headers
    }

}
