package ru.yandex.direct.web.entity.smsauth.service;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.smsauth.model.SmsAuthKey;
import ru.yandex.direct.core.smsauth.service.SmsAuthStorageService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.smsauth.model.SmsAuthBaseInternalRequest;
import ru.yandex.direct.web.entity.smsauth.model.checkpassword.CheckPasswordInternalRequest;
import ru.yandex.direct.web.entity.smsauth.model.sendsms.SendSmsByPhoneIdInternalRequest;

import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicateOfNullable;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.web.entity.smsauth.validation.SmsAuthDefects.incorrectPassword;
import static ru.yandex.direct.web.entity.smsauth.validation.SmsAuthDefects.incorrectUserPhoneId;
import static ru.yandex.direct.web.entity.smsauth.validation.SmsAuthDefects.noRetriesLeft;
import static ru.yandex.direct.web.entity.smsauth.validation.SmsAuthDefects.sessionAlreadyLocked;
import static ru.yandex.direct.web.entity.smsauth.validation.SmsAuthDefects.smsLimitExceeded;
import static ru.yandex.direct.web.entity.smsauth.validation.SmsAuthDefects.smsNotSent;

@Service
@ParametersAreNonnullByDefault
public class SmsAuthValidationService {

    private final UserService userService;
    private final SmsAuthStorageService smsAuthStorageService;

    @Autowired
    public SmsAuthValidationService(UserService userService, SmsAuthStorageService smsAuthStorageService) {
        this.userService = userService;
        this.smsAuthStorageService = smsAuthStorageService;
    }

    /**
     * Превалидация нужна, чтобы локать сессию в ручках, использующихся для аутентификации по смс.
     * Лок сессии позволяет нам избежать параллельных запросов, которые могут быть потенциально опасны.
     * Если дернуть ручку для залоченной сессии - вернется ошибка валидации.
     */
    public ValidationResult<SmsAuthBaseInternalRequest, Defect> preValidate(SmsAuthBaseInternalRequest request) {
        ItemValidationBuilder<SmsAuthBaseInternalRequest, Defect> vb = ItemValidationBuilder.of(request);

        vb.item(request.getOperatorUid(), "operatorUid")
                .check(canLockSession(request));

        return vb.getResult();
    }

    public ValidationResult<SendSmsByPhoneIdInternalRequest, Defect> validate(SendSmsByPhoneIdInternalRequest request) {
        ItemValidationBuilder<SendSmsByPhoneIdInternalRequest, Defect> vb = ItemValidationBuilder.of(request);

        vb.item(request.getPhoneId(), "phoneId")
                .check(notNull())
                .check(correctUserPhoneId(request), When.isValid());

        // Если есть ошибки валидации на данном этапе - сразу выходим, чтобы не инкрементить ему количество
        // отправленных смс
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        vb.item(request.getOperatorUid(), "operatorUid")
                .check(userCanSendSms(), When.isValid());

        return vb.getResult();
    }

    public ValidationResult<CheckPasswordInternalRequest, Defect> validate(CheckPasswordInternalRequest request) {
        ItemValidationBuilder<CheckPasswordInternalRequest, Defect> vb = ItemValidationBuilder.of(request);

        vb.item(request.getOperatorUid(), "operatorUid")
                .check(smsIsSent(request))
                .check(userCanTryAgain(request), When.isValid());

        // Выходим, чтобы вернуть одну ошибку, если пользователь, помимо прочего, еще и пароль неправильно ввел
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        vb.item(request.getPassword(), "password")
                .check(notNull())
                .check(correctPassword(request), When.isValid());

        return vb.getResult();
    }

    // Constraints

    /**
     * Констреинт, проверяющий что пользователь указал корректный телефон для отправки смс.
     *
     * @param request   запрос аутентификации через смс
     */
    private Constraint<Long, Defect> correctUserPhoneId(SendSmsByPhoneIdInternalRequest request) {
        Long smsPhoneId = userService.getSmsPhoneId(request.getOperatorUid());

        return fromPredicateOfNullable(requestPhoneId -> Objects.equals(requestPhoneId, smsPhoneId),
                incorrectUserPhoneId());
    }

    /**
     * Констреинт, проверяющий что у пользователь еще не исчерпал суточный лимит отправляемых смс.
     */
    private Constraint<Long, Defect> userCanSendSms() {
        return fromPredicate(smsAuthStorageService::checkUserCanSendSms, smsLimitExceeded());
    }

    /**
     * Констреинт, проверяющий можно ли залочить сессию. Если можно - сразу же локает ее.
     *
     * @param request   запрос аутентификации через смс
     */
    private Constraint<Long, Defect> canLockSession(SmsAuthBaseInternalRequest request) {
        return fromPredicate(operatorUid -> {
            String sessionIdHash = request.getSessionIdHash();
            SmsAuthKey key = SmsAuthKey.ofKeyParts(operatorUid, sessionIdHash);

            return smsAuthStorageService.checkAndLockSession(key);
        }, sessionAlreadyLocked());
    }

    /**
     * Констреинт, проверяющий было ли отправлено смс с кодом в текущей сессии.
     *
     * @param request   запрос попытки ввода пароля
     */
    private Constraint<Long, Defect> smsIsSent(CheckPasswordInternalRequest request) {
        return fromPredicate(operatorUid -> {
            String sessionIdHash = request.getSessionIdHash();
            SmsAuthKey key = SmsAuthKey.ofKeyParts(operatorUid, sessionIdHash);

            return smsAuthStorageService.checkSmsIsSent(key);
        }, smsNotSent());
    }

    /**
     * Констреинт, проверяющий что пользователь еще не исчерпал попытки ввода пароля.
     *
     * @param request   запрос попытки ввода пароля
     */
    private Constraint<Long, Defect> userCanTryAgain(CheckPasswordInternalRequest request) {
        return fromPredicate(operatorUid -> {
            String sessionIdHash = request.getSessionIdHash();
            SmsAuthKey key = SmsAuthKey.ofKeyParts(operatorUid, sessionIdHash);

            return smsAuthStorageService.userCanTryAgain(key);
        }, noRetriesLeft());
    }

    /**
     * Констреинт, проверяющий что пользователь ввел правильный пароль.
     *
     * @param request   запрос попытки ввода пароля
     */
    private Constraint<String, Defect> correctPassword(CheckPasswordInternalRequest request) {
        return fromPredicate(password -> {
            Long operatorUid = request.getOperatorUid();
            String sessionIdHash = request.getSessionIdHash();
            SmsAuthKey key = SmsAuthKey.ofKeyParts(operatorUid, sessionIdHash);

            return smsAuthStorageService.checkPassword(key, sessionIdHash, password);
        }, incorrectPassword());
    }
}
