package ru.yandex.direct.web.entity.stat.controller;

import java.io.IOException;
import java.util.List;
import java.util.Locale;

import javax.servlet.http.HttpServletResponse;

import com.google.common.net.HttpHeaders;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.apache.commons.httpclient.HttpStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.mdsfile.service.MdsFileService;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.stat.service.ConversionStatService;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.DateTimeUtils.today;
import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;
import static ru.yandex.direct.web.entity.excel.service.validation.ExcelConstraints.XLS_CONTENT_TYPE;

@Controller
@RequestMapping(value = "/stat")
@Api(tags = "stat")
public class StatController {
    private static final String CSV_TYPE = "text/csv;charset=UTF-8";
    private static final String CONVERSION_REPORT_ENDPOINT = "conversion_csv_report";
    private static final String BRAND_SAFETY_REPORT_ENDPOINT = "brand_safety_excel_report";

    private final DirectWebAuthenticationSource authenticationSource;
    private final ConversionStatService conversionStatService;
    private final MdsFileService mdsFileService;

    @Autowired
    public StatController(
            DirectWebAuthenticationSource authenticationSource,
            ConversionStatService conversionStatService,
            MdsFileService mdsFileService
    ) {
        this.authenticationSource = authenticationSource;
        this.conversionStatService = conversionStatService;
        this.mdsFileService = mdsFileService;
    }

    @ApiOperation(
            value = "conversionCsvReport",
            httpMethod = "GET",
            nickname = "conversionCsvReport"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params"),
                    @ApiResponse(code = 403, message = "Permission denied"),
            }
    )
    @PreAuthorizeRead
    @GetMapping(value = CONVERSION_REPORT_ENDPOINT, produces = CSV_TYPE)
    @ResponseBody
    // todo: эта ручка в данный момент работает некорректно — она не должна дёргаться напрямую клиентом (DIRECT-126472)
    public void getConversionCsvReport(
            @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin,
            @RequestParam(name = "campaign_id", required = false) Long campaignId,
            @RequestParam(value = "lang", required = false) String language,
            HttpServletResponse response) throws IOException {
        DirectAuthentication auth = authenticationSource.getAuthentication();
        ClientId clientId = auth.getSubjectUser().getClientId();
        var operatorUid = auth.getOperator().getUid();

        response.setContentType(CSV_TYPE);
        response.setCharacterEncoding("UTF-8");
        String fileNamePrefix = campaignId == null ? auth.getSubjectUser().getLogin() : campaignId.toString();
        response.setHeader(HttpHeaders.CONTENT_DISPOSITION,
                String.format("attachment; filename=\"%s_%s_conversions_report.csv\"", fileNamePrefix, today()));

        var locale = ifNotNull(language, Locale::new);

        if (!conversionStatService.writeReport(response.getOutputStream(), operatorUid, clientId, campaignId, locale)) {
            response.setStatus(HttpServletResponse.SC_NOT_FOUND);
        }
    }

    @ApiOperation(
            value = "brandSafetyExcelReport",
            httpMethod = "GET",
            nickname = "brandSafetyExcelReport"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 404, message = "Not Found"),
            }
    )
    @PreAuthorizeRead
    @GetMapping(value = BRAND_SAFETY_REPORT_ENDPOINT, produces = XLS_CONTENT_TYPE)
    @ResponseBody
    public void brandSafetyExcelReport(
            @RequestParam(value = "file_key") String fileKey,
            @RequestParam(value = PARAMETER_ULOGIN) String subjectLogin,
            HttpServletResponse response) {
        var auth = authenticationSource.getAuthentication();
        var clientId = auth.getSubjectUser().getClientId();

        var metaData = mdsFileService.getMetaData(clientId, List.of(fileKey)).get(fileKey);
        if (metaData == null) {
            response.setStatus(HttpStatus.SC_NOT_FOUND);
            return;
        }

        var mdsId = metaData.getId();
        var customName = mdsFileService.getCustomNames(clientId, List.of(mdsId)).get(mdsId);

        HttpUtil.setInternalRedirectToMdsDirectFiles(response, mdsFileService.getUrl(metaData), customName);
    }
}
