package ru.yandex.direct.web.entity.transparencyreport.controller;

import java.time.LocalDate;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.captcha.CaptchaConditionKey;
import ru.yandex.direct.web.core.security.captcha.CaptchaLimits;
import ru.yandex.direct.web.core.security.captcha.CaptchaSecured;
import ru.yandex.direct.web.core.security.captcha.DisableAutoCaptcha;
import ru.yandex.direct.web.core.security.csrf.CsrfCheck;
import ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportGetBannersRequest;
import ru.yandex.direct.web.entity.transparencyreport.service.TransparencyReportService;

import static ru.yandex.direct.web.core.security.configuration.BlackboxWebAuthenticationConfiguration.PUBLIC_BASE_URL;

/**
 * Публичный контроллер для страницы с отчетом о прозрачности.
 */
@Controller
@RequestMapping(value = PUBLIC_BASE_URL + "transparency-report", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class TransparencyReportController {
    private final TransparencyReportService transparencyReportService;

    @Autowired
    public TransparencyReportController(TransparencyReportService transparencyReportService) {
        this.transparencyReportService = transparencyReportService;
    }

    @DisableAutoCaptcha
    @CsrfCheck(enabled = false)
    @CaptchaSecured(
            keys = CaptchaConditionKey.IP_VALUE,
            limits = @CaptchaLimits(freq = 50, interval = 3600, maxFreq = 4000))
    @RequestMapping(path = "/api/v1/campaigns/creatives", method = RequestMethod.GET)
    @ResponseBody
    public WebResponse getCreatives(
            @RequestParam(name = "client_name", required = false) String clientName,
            @RequestParam(name = "date_from", required = false) @DateTimeFormat(iso = DateTimeFormat.ISO.DATE) LocalDate dateFrom,
            @RequestParam(name = "date_to", required = false) @DateTimeFormat(iso = DateTimeFormat.ISO.DATE) LocalDate dateTo,
            @RequestParam(name = "from_banner_id", required = false) String bannerId,
            @RequestParam(name = "limit", required = false) Integer limit,
            @RequestParam(name = "money_from", required = false) Long moneyFrom,
            @RequestParam(name = "money_to", required = false) Long moneyTo,
            @RequestParam(name = "creative_type", required = false) String creativeType)
    {
        TransparencyReportGetBannersRequest request = new TransparencyReportGetBannersRequest()
                .withClientName(clientName)
                .withDateFrom(dateFrom)
                .withDateTo(dateTo)
                .withFromBannerId(bannerId)
                .withLimit(limit)
                .withMoneyFrom(moneyFrom)
                .withMoneyTo(moneyTo)
                .withCreativeType(creativeType);

        return transparencyReportService.getBanners(request);
    }

    @DisableAutoCaptcha
    @CsrfCheck(enabled = false)
    @CaptchaSecured(
            keys = CaptchaConditionKey.IP_VALUE,
            limits = @CaptchaLimits(freq = 50, interval = 3600, maxFreq = 4000))
    @RequestMapping(path = "/api/v1/clients", method = RequestMethod.GET)
    @ResponseBody
    public WebResponse getClients() {
        return transparencyReportService.getClients();
    }
}
