package ru.yandex.direct.web.entity.transparencyreport.service;

import java.text.SimpleDateFormat;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import org.springframework.stereotype.Service;

import ru.yandex.clickhouse.response.ClickHouseResponse;
import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.entity.transparencyreport.exception.TransparencyReportParseMinMaxDateException;
import ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportBanner;
import ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportBannersWithSummary;
import ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportClient;
import ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportClients;
import ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportGetBannersRequest;
import ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportGetBannersResponse;
import ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportGetClientsResponse;
import ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportMinMaxDate;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_BANNER_BODY;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_BANNER_DATE_END;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_BANNER_DATE_START;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_BANNER_HREF;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_BANNER_ID;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_BANNER_IS_GRANT;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_BANNER_TITLE;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_CLIENT_HASH;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_CREATIVE_HEIGHT;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_CREATIVE_ID;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_CREATIVE_LIVE_PREVIEW_URL;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_CREATIVE_PREVIEW_URL;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_CREATIVE_TYPE;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_CREATIVE_WIDTH;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_COLUMN_IMAGE_DATA;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_STATS_COLUMN_BANNER_ID;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_STATS_COLUMN_DT;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_STATS_COLUMN_REVENUE;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_STATS_COLUMN_REVENUE_SUM_ALIAS;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_STATS_COLUMN_SHOWS;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_STATS_COLUMN_SHOWS_SUM_ALIAS;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_STATS_TABLE_NAME;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.BANNERS_TABLE_NAME;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.CLIENTS_COLUMN_CLIENT_HASH;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.CLIENTS_COLUMN_CLIENT_NAME;
import static ru.yandex.direct.web.entity.transparencyreport.model.TransparencyReportDbInfo.CLIENTS_TABLE_NAME;

@Service
@ParametersAreNonnullByDefault
public class TransparencyReportService {
    private static final int GET_BANNERS_LIMIT_DEFAULT = 200;
    private static final int GET_CLIENTS_LIMIT = 2000;

    private final DatabaseWrapperProvider dbProvider;
    private final TransparencyReportValidationService transparencyReportValidationService;
    private final ValidationResultConversionService validationResultConversionService;
    private final Supplier<TransparencyReportMinMaxDate> minMaxDateCache;
    private final Supplier<List<TransparencyReportClient>> clientsCache;

    public TransparencyReportService(DatabaseWrapperProvider dbProvider,
                                     TransparencyReportValidationService transparencyReportValidationService,
                                     ValidationResultConversionService validationResultConversionService) {
        this.dbProvider = dbProvider;
        this.transparencyReportValidationService = transparencyReportValidationService;
        this.validationResultConversionService = validationResultConversionService;
        this.minMaxDateCache = Suppliers.memoizeWithExpiration(this::getMinMaxDate, 10, TimeUnit.MINUTES);
        this.clientsCache = Suppliers.memoizeWithExpiration(this::getClientsFromDb, 10, TimeUnit.MINUTES);
    }

    /**
     * Получаем данные о креативах для отчета о прозрачности.
     */
    public WebResponse getBanners(TransparencyReportGetBannersRequest request) {
        ValidationResult<TransparencyReportGetBannersRequest, Defect> validation =
                transparencyReportValidationService.validateGetBannersRequest(request);

        if (validation.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(validation);
        }

        SqlBuilder builder = new SqlBuilder();

        builder.selectExpression(
                String.format("%s.%s", BANNERS_TABLE_NAME, BANNERS_COLUMN_BANNER_ID),
                BANNERS_COLUMN_BANNER_ID);
        builder.select(CLIENTS_COLUMN_CLIENT_NAME);
        builder.select(BANNERS_COLUMN_CREATIVE_ID);
        builder.select(BANNERS_COLUMN_CREATIVE_TYPE);
        builder.select(BANNERS_COLUMN_CREATIVE_LIVE_PREVIEW_URL);
        builder.select(BANNERS_COLUMN_CREATIVE_PREVIEW_URL);
        builder.select(BANNERS_COLUMN_BANNER_DATE_START);
        builder.select(BANNERS_COLUMN_BANNER_DATE_END);
        builder.select(BANNERS_COLUMN_CREATIVE_HEIGHT);
        builder.select(BANNERS_COLUMN_CREATIVE_WIDTH);
        builder.select(BANNERS_COLUMN_BANNER_TITLE);
        builder.select(BANNERS_COLUMN_BANNER_BODY);
        builder.select(BANNERS_COLUMN_BANNER_HREF);
        builder.select(BANNERS_COLUMN_BANNER_IS_GRANT);
        builder.select(BANNERS_COLUMN_IMAGE_DATA);
        builder.select(BANNERS_STATS_COLUMN_REVENUE_SUM_ALIAS);
        builder.select(BANNERS_STATS_COLUMN_SHOWS_SUM_ALIAS);

        String bannerStatsExpression = buildBannersStatsExpression(request);

        builder.from(BANNERS_TABLE_NAME);
        builder.join(String.format("%s ON %s.%s = %s.%s", CLIENTS_TABLE_NAME, BANNERS_TABLE_NAME,
                BANNERS_COLUMN_CLIENT_HASH, CLIENTS_TABLE_NAME, CLIENTS_COLUMN_CLIENT_HASH));
        builder.join(String.format("(%s) AS %s ON %s.%s = %s.%s",
                bannerStatsExpression, BANNERS_STATS_TABLE_NAME, BANNERS_TABLE_NAME, BANNERS_COLUMN_BANNER_ID,
                BANNERS_STATS_TABLE_NAME, BANNERS_STATS_COLUMN_BANNER_ID));

        if (request.getClientName() != null) {
            builder.where(String.format("match(%s, ?)", CLIENTS_COLUMN_CLIENT_NAME),
                    "(?i)" + request.getClientName().strip());
        }
        if (request.getCreativeType() != null) {
            builder.where(BANNERS_COLUMN_CREATIVE_TYPE + " = ?", request.getCreativeType());
        }
        if (request.getFromBannerId() != null) {
            String selectBannerDateStartExpression = new SqlBuilder()
                    .select(BANNERS_COLUMN_BANNER_DATE_START)
                    .from(BANNERS_TABLE_NAME)
                    .where(String.format("%s = '%s'", BANNERS_COLUMN_BANNER_ID, request.getFromBannerId()))
                    .toString();
            builder.where(
                    String.format("(%s = (%s) AND %s < ? OR %s < (%s))",
                            BANNERS_COLUMN_BANNER_DATE_START, selectBannerDateStartExpression, BANNERS_COLUMN_BANNER_ID,
                            BANNERS_COLUMN_BANNER_DATE_START, selectBannerDateStartExpression),
                    request.getFromBannerId());
        }

        builder.orderBy(BANNERS_COLUMN_BANNER_DATE_START, SqlBuilder.Order.DESC);
        builder.orderBy(BANNERS_COLUMN_BANNER_ID, SqlBuilder.Order.DESC);

        int limit = request.getLimit() != null ? request.getLimit() : GET_BANNERS_LIMIT_DEFAULT;
        builder.limit(limit);

        List<TransparencyReportBanner> banners = dbProvider.get(SimpleDb.CLICKHOUSE_TRANSP_REPORT).query(
                builder.toString(), builder.getBindings(), new TransparencyReportBannersRowMapper());

        TransparencyReportMinMaxDate minMaxDate = minMaxDateCache.get();

        TransparencyReportBannersWithSummary bannersWithSummary = new TransparencyReportBannersWithSummary()
                .withBanners(banners)
                .withMinDateAvailable(minMaxDate.getMinDate())
                .withMaxDateAvailable(minMaxDate.getMaxDate())
                .withIsEnd(banners.size() < limit);

        return new TransparencyReportGetBannersResponse()
                .withResult(bannersWithSummary);
    }

    public WebResponse getClients() {
        List<TransparencyReportClient> clients = clientsCache.get();

        TransparencyReportClients clientsResult = new TransparencyReportClients()
                .withClients(clients);

        return new TransparencyReportGetClientsResponse()
                .withResult(clientsResult);
    }

    /**
     * Получаем данные о всех клиентах из бд для отчета о прозрачности.
     */
    private List<TransparencyReportClient> getClientsFromDb() {
        SqlBuilder builder = new SqlBuilder();

        builder.selectExpression(
                String.format("any(%s)", CLIENTS_COLUMN_CLIENT_NAME),
                CLIENTS_COLUMN_CLIENT_NAME);
        builder.selectExpression(
                String.format("sum(%s)", BANNERS_STATS_COLUMN_REVENUE_SUM_ALIAS),
                BANNERS_STATS_COLUMN_REVENUE_SUM_ALIAS);
        builder.selectExpression(
                String.format("sum(%s)", BANNERS_STATS_COLUMN_SHOWS_SUM_ALIAS),
                BANNERS_STATS_COLUMN_SHOWS_SUM_ALIAS);

        String bannerStatsExpression = buildBannersStatsExpression(new TransparencyReportGetBannersRequest());

        builder.from(BANNERS_TABLE_NAME);
        builder.join(String.format("%s ON %s.%s = %s.%s", CLIENTS_TABLE_NAME, BANNERS_TABLE_NAME,
                BANNERS_COLUMN_CLIENT_HASH, CLIENTS_TABLE_NAME, CLIENTS_COLUMN_CLIENT_HASH));
        builder.join(String.format("(%s) AS %s ON %s.%s = %s.%s",
                bannerStatsExpression, BANNERS_STATS_TABLE_NAME, BANNERS_TABLE_NAME, BANNERS_COLUMN_BANNER_ID,
                BANNERS_STATS_TABLE_NAME, BANNERS_STATS_COLUMN_BANNER_ID));

        builder.groupByExpression(BANNERS_TABLE_NAME + "." + BANNERS_COLUMN_CLIENT_HASH);

        builder.orderBy(BANNERS_STATS_COLUMN_REVENUE_SUM_ALIAS, SqlBuilder.Order.DESC);
        builder.orderBy(CLIENTS_COLUMN_CLIENT_NAME, SqlBuilder.Order.ASC);

        builder.limit(GET_CLIENTS_LIMIT);

        return dbProvider.get(SimpleDb.CLICKHOUSE_TRANSP_REPORT).query(
                builder.toString(), builder.getBindings(), new TransparencyReportClientsRowMapper());
    }

    /**
     * Получаем минимальную и максимальную даты, за которые в бд присутствуют записи.
     */
    private TransparencyReportMinMaxDate getMinMaxDate() {
        SqlBuilder builder = new SqlBuilder()
                .selectExpression("min(dt)", "min_date")
                .selectExpression("max(dt)", "max_date")
                .from(BANNERS_STATS_TABLE_NAME);

        ClickHouseResponse response = dbProvider.get(SimpleDb.CLICKHOUSE_TRANSP_REPORT).clickhouseQuery(
                builder.toString(), builder.getBindings());

        try {
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
            return new TransparencyReportMinMaxDate()
                    .withMinDate(formatter.parse(response.getData().get(0).get(0)))
                    .withMaxDate(formatter.parse(response.getData().get(0).get(1)));
        } catch (Exception e) {
            throw new TransparencyReportParseMinMaxDateException("can't parse min max date from db", e);
        }
    }

    /**
     * Подзапрос для получения статистики по баннерам, джойнится с таблицей баннеров
     */
    private static String buildBannersStatsExpression(TransparencyReportGetBannersRequest request) {
        SqlBuilder bannerStatsBuilder = new SqlBuilder();

        bannerStatsBuilder.select(BANNERS_STATS_COLUMN_BANNER_ID);
        bannerStatsBuilder.selectExpression(
                String.format("sum(%s)", BANNERS_STATS_COLUMN_REVENUE),
                BANNERS_STATS_COLUMN_REVENUE_SUM_ALIAS);
        bannerStatsBuilder.selectExpression(
                String.format("sum(%s)", BANNERS_STATS_COLUMN_SHOWS),
                BANNERS_STATS_COLUMN_SHOWS_SUM_ALIAS);

        bannerStatsBuilder.from(BANNERS_STATS_TABLE_NAME);

        if (request.getDateFrom() != null) {
            bannerStatsBuilder.where(String.format("%s >= '%s'", BANNERS_STATS_COLUMN_DT,
                    DateTimeFormatter.ISO_LOCAL_DATE.format(request.getDateFrom())));
        }
        if (request.getDateTo() != null) {
            bannerStatsBuilder.where(String.format("%s <= '%s'", BANNERS_STATS_COLUMN_DT,
                    DateTimeFormatter.ISO_LOCAL_DATE.format(request.getDateTo())));
        }
        if (request.getMoneyFrom() != null) {
            bannerStatsBuilder.having(BANNERS_STATS_COLUMN_REVENUE_SUM_ALIAS + " >= " + request.getMoneyFrom());
        }
        if (request.getMoneyTo() != null) {
            bannerStatsBuilder.having(BANNERS_STATS_COLUMN_REVENUE_SUM_ALIAS + " <= " + request.getMoneyTo());
        }
        bannerStatsBuilder.groupBy(BANNERS_STATS_COLUMN_BANNER_ID);

        return bannerStatsBuilder.toString();
    }
}
