package ru.yandex.direct.web.entity.uac.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import org.springframework.http.HttpStatus
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RestController
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder
import ru.yandex.direct.core.entity.uac.service.UacDbDefineService
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiAggregatedShowConditionOrderBy
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiAggregatedShowConditionOrderByField
import ru.yandex.direct.grid.model.Order
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN
import ru.yandex.direct.web.entity.uac.notOwnerResponse
import ru.yandex.direct.web.entity.uac.service.UacAggregatedShowConditionsService
import ru.yandex.direct.web.entity.uac.toResponse

@RestController
@Api(tags = ["uac"])
@RequestMapping("/uac", produces = [MediaType.APPLICATION_JSON_VALUE])
class UacAggregatedShowConditionsController(
    private val uacAggregatedShowConditionsService: UacAggregatedShowConditionsService,
    private val authenticationSource: DirectWebAuthenticationSource,
    private val uacDbDefineService: UacDbDefineService,
    private val serviceHolder: UacCampaignServiceHolder
) {
    @ApiOperation(
        value = "getShowConditionsStat",
        httpMethod = "GET",
        nickname = "getShowConditionsStat"
    )
    @PreAuthorizeRead
    @GetMapping(
        value = ["/show_conditions_stat"],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    fun getShowConditions(
        @RequestParam(value = "campaign_id", required = true) directCampaignId: Long,
        @RequestParam(value = "from", required = true) fromSeconds: Long,
        @RequestParam(value = "to", required = true) toSeconds: Long,
        @RequestParam(value = "order", required = false) order: Order?,
        @RequestParam(value = "by_field", required = false) byField: GdiAggregatedShowConditionOrderByField?,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?
    ): ResponseEntity<Any> {
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser

        val useGrut = uacDbDefineService.useGrutForDirectCampaignId(directCampaignId)
        val uacCampaignService = serviceHolder.getUacCampaignService(useGrut)

        val isVisible = uacCampaignService.checkVisibleCampaign(
            operator.uid,
            subjectUser.clientId,
            directCampaignId
        )
        if (!isVisible) {
            return notOwnerResponse()
        }

        val orderBy: GdiAggregatedShowConditionOrderBy? = if (byField == null) null
            else GdiAggregatedShowConditionOrderBy()
                .withOrder(order ?: Order.ASC)
                .withField(byField)

        val aggregatedShowConditions = uacAggregatedShowConditionsService.getAggregatedShowConditions(
            operator.uid,
            subjectUser.clientId,
            directCampaignId,
            fromSeconds,
            toSeconds,
            orderBy
        )
        return ResponseEntity(
            aggregatedShowConditions.toResponse(),
            HttpStatus.OK
        )
    }
}
