package ru.yandex.direct.web.entity.uac.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiParam
import org.slf4j.LoggerFactory
import org.springframework.http.HttpStatus
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RestController
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.utils.model.UrlParts
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider
import ru.yandex.direct.web.entity.uac.invalidUrl
import ru.yandex.direct.web.entity.uac.model.UacQr
import ru.yandex.direct.web.entity.uac.notFoundResponse
import ru.yandex.direct.web.entity.uac.service.UacMobileAppService
import ru.yandex.direct.web.entity.uac.toResponse

@RestController
@Api(tags = ["uac"])
@RequestMapping("/uac", produces = [MediaType.APPLICATION_JSON_VALUE])
open class UacAppInfoController(
    private val authenticationSource: DirectWebAuthenticationSource,
    private val uacMobileAppService: UacMobileAppService,
    private val serviceHolder: UacCampaignServiceHolder,
    private val grutTransactionProvider: GrutTransactionProvider,
) {
    companion object {
        private val logger = LoggerFactory.getLogger(UacAppMetrikaAppsController::class.java)
    }

    @ApiOperation(
        value = "qrAppInfoGet",
        httpMethod = "GET",
        nickname = "qrAppInfoGet",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["app_info/qr"])
    open fun getQrApp(
        @ApiParam(value = "app.bundle_id", required = true) @RequestParam(value = "bundle_id") bundleId: String,
        @ApiParam(value = "count-link", required = true) @RequestParam(value = "count_link") countLink: String
    ): ResponseEntity<Any> {
        val appInfos = uacMobileAppService.getAppInfoByBundleIds(listOf(bundleId))
        if (appInfos.isEmpty()) {
            return notFoundResponse()
        }
        val appInfo = uacMobileAppService.getAppInfo(appInfos[0].id) ?: return notFoundResponse()

        return try {
            ResponseEntity(UacQr(appInfo, addIsQrToCountLink(countLink)).toResponse(), HttpStatus.OK)
        } catch (e: IllegalArgumentException) {
            logger.error("Invalid count-link: $countLink")
            invalidUrl()
        }
    }

    private fun addIsQrToCountLink(countLink: String): String {
        val urlParts = UrlParts.fromUrl(countLink)
        val params = urlParts.parameters
        val builder = urlParts.toBuilder()
            .withParameters(params)
        builder.replaceOrAddParam("is_qr", "1")
        return builder.build().toUrl()
    }

    @ApiOperation(
        value = "myApps",
        httpMethod = "GET",
        nickname = "myApps",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["app_info/my_apps"])
    open fun getMyApps(
        @RequestParam(value = DirectCookieAuthProvider.PARAMETER_ULOGIN) ulogin: String?
    ): ResponseEntity<Any> {
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser

        val accountId = serviceHolder.getUacCampaignService(false).getOrCreateClient(operator, subjectUser)
        val appIdsYdb = getAppIds(useGrut = false, accountId)

        val accountIdGrut = serviceHolder.getUacCampaignService(true).getOrCreateClient(operator, subjectUser)
        val appIdsGrut = grutTransactionProvider.runInTransaction { getAppIds(useGrut = true, accountIdGrut) }

        val appInfos = uacMobileAppService.getAppInfos(appIdsYdb union appIdsGrut)
            .sortedBy { it.title?.lowercase() ?: "" }

        return ResponseEntity(appInfos.toResponse(), HttpStatus.OK)
    }

    private fun getAppIds(useGrut: Boolean, accountId: String): List<String> {
        val uacCampaignService = serviceHolder.getUacCampaignService(useGrut)
        return uacCampaignService.getAppIdsByAccountId(accountId)
    }
}
