package ru.yandex.direct.web.entity.uac.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiParam
import org.slf4j.LoggerFactory
import org.springframework.http.HttpStatus
import org.springframework.http.MediaType.APPLICATION_JSON_VALUE
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RestController
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN
import ru.yandex.direct.web.entity.uac.model.UacAppInfosResponse
import ru.yandex.direct.web.entity.uac.model.UacCreateTrackingUrlResponse
import ru.yandex.direct.web.entity.uac.notFoundResponse
import ru.yandex.direct.web.entity.uac.service.UacAppMetrikaService
import ru.yandex.direct.web.entity.uac.service.UacMobileAppService
import ru.yandex.direct.web.entity.uac.service.UacSuggestService
import ru.yandex.direct.web.entity.uac.toResponse

@RestController
@Api(tags = ["uac"])
@RequestMapping("/uac", produces = [APPLICATION_JSON_VALUE])
open class UacAppMetrikaAppsController(
    private val authenticationSource: DirectWebAuthenticationSource,
    private val uacSuggestService: UacSuggestService,
    private val uacMobileAppService: UacMobileAppService,
    private val uacAppMetrikaService: UacAppMetrikaService,
) {
    companion object {
        private val logger = LoggerFactory.getLogger(UacAppMetrikaAppsController::class.java)
    }

    @ApiOperation(
        value = "appMetrikaAppsGet",
        httpMethod = "GET",
        nickname = "appMetrikaAppsGet",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["appmetrika/applications"])
    open fun getAppMetricaApps(
        @ApiParam(value = "text", required = false) @RequestParam text: String?,
        @ApiParam(value = PARAMETER_ULOGIN, required = false) @RequestParam ulogin: String?
    ): ResponseEntity<UacAppInfosResponse> {
        val subjectUser = authenticationSource.authentication.subjectUser
        authenticationSource.authentication.operator

        val apps = uacSuggestService.suggestAppMetrikaApps(subjectUser, text)

        return ResponseEntity<UacAppInfosResponse>(
            UacAppInfosResponse(Trace.current().spanId.toString()).withResult(apps),
            null,
            HttpStatus.OK
        )
    }

    @ApiOperation(
        value = "appMetrikaTrackingUrlCreate",
        httpMethod = "POST",
        nickname = "appMetrikaTrackingUrlCreate",
    )
    @PreAuthorizeRead
    @PostMapping(value = ["appmetrika/tracking_url"])
    open fun createAppMetrikaUrl(
        @ApiParam(value = "app_info_id", required = true) @RequestParam("app_info_id") appInfoId: String?,
        @ApiParam(value = PARAMETER_ULOGIN, required = false) @RequestParam ulogin: String?
    ): ResponseEntity<Any> {
        val subjectUser = authenticationSource.authentication.subjectUser
        authenticationSource.authentication.operator

        val appInfo = uacMobileAppService.getAppInfo(appInfoId) ?: return notFoundResponse()
        val createTrackingUrlResponse =
            uacAppMetrikaService.createAppMetrikaUrl(appInfo, subjectUser) ?: return notFoundResponse()

        return ResponseEntity(createTrackingUrlResponse.toResponse(), HttpStatus.OK)
    }
}
