package ru.yandex.direct.web.entity.uac.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Value
import org.springframework.http.HttpHeaders
import org.springframework.http.HttpStatus
import org.springframework.http.MediaType.APPLICATION_JSON_VALUE
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.WebDataBinder
import org.springframework.web.bind.annotation.DeleteMapping
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.InitBinder
import org.springframework.web.bind.annotation.PatchMapping
import org.springframework.web.bind.annotation.PathVariable
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestHeader
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RestController
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.db.PpcPropertyNames.CHANGE_REJECTED_ASSET_TO_CREATED_WITH_SECONDS_SHIFT
import ru.yandex.direct.common.db.PpcPropertyNames.UAC_GET_CAMPAIGN_PROTO_RESPONSE_DIFF_PERCENT
import ru.yandex.direct.core.entity.adgeneration.ZenMetaInfoService
import ru.yandex.direct.core.entity.banner.service.BannerService
import ru.yandex.direct.core.entity.calltrackingsettings.service.CalltrackingSettingsService
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.uac.getCurrentLanguage
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.isValidId
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.CampaignStatuses
import ru.yandex.direct.core.entity.uac.model.ContentStatisticsOrderBy
import ru.yandex.direct.core.entity.uac.model.ContentStatisticsSortBy
import ru.yandex.direct.core.entity.uac.model.DirectCampaignStatus
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.Status
import ru.yandex.direct.core.entity.uac.model.TargetStatus
import ru.yandex.direct.core.entity.uac.model.UacGoal
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.service.AudienceSegmentsService
import ru.yandex.direct.core.entity.uac.service.BaseUacCampaignService
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder
import ru.yandex.direct.core.entity.uac.service.UacDbDefineService
import ru.yandex.direct.core.entity.uac.service.UacRejectReasonService
import ru.yandex.direct.core.entity.uac.service.stat.AssetStatService
import ru.yandex.direct.core.entity.uac.service.stat.ContentEfficiencyWithExistenceAndStat
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.core.validation.defects.Defects.metrikaReturnsResultWithErrors
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.metrika.client.MetrikaClient
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.result.Result
import ru.yandex.direct.utils.CommonUtils.nvl
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN
import ru.yandex.direct.web.entity.uac.actionNotAllowedResponse
import ru.yandex.direct.web.entity.uac.campaignAlreadyStartedResponse
import ru.yandex.direct.web.entity.uac.campaignAlreadyStoppedResponse
import ru.yandex.direct.web.entity.uac.campaignDoesNotExistResponse
import ru.yandex.direct.web.entity.uac.campaignIsArchivedResponse
import ru.yandex.direct.web.entity.uac.cantStartCpmBannerCampaign
import ru.yandex.direct.web.entity.uac.converter.UacContentStatisticsConverter.toContentStatistics
import ru.yandex.direct.web.entity.uac.converter.UacContentStatisticsConverter.toCreativeStatistics
import ru.yandex.direct.web.entity.uac.errorCreatingDirectCampaignResponse
import ru.yandex.direct.web.entity.uac.model.CampaignStatusesRequest
import ru.yandex.direct.web.entity.uac.model.CreateCampaignRequest
import ru.yandex.direct.web.entity.uac.model.CreatingGoalType
import ru.yandex.direct.web.entity.uac.model.CreativeStatistics
import ru.yandex.direct.web.entity.uac.model.GoalCreateRequest
import ru.yandex.direct.web.entity.uac.model.PatchCampaignInternalRequest
import ru.yandex.direct.web.entity.uac.model.PatchCampaignRequest
import ru.yandex.direct.web.entity.uac.model.UacNavigationInfo
import ru.yandex.direct.web.entity.uac.notAllowedResponse
import ru.yandex.direct.web.entity.uac.notFoundResponse
import ru.yandex.direct.web.entity.uac.notOwnerResponse
import ru.yandex.direct.web.entity.uac.service.BaseUacCampaignAddService.Companion.UAC_PATH_NODE_CONVERTER_PROVIDER
import ru.yandex.direct.web.entity.uac.service.UacCampaignWebServiceHolder
import ru.yandex.direct.web.entity.uac.service.UacCpmStatisticsService
import ru.yandex.direct.web.entity.uac.service.history.UacCampaignHistoryService
import ru.yandex.direct.web.entity.uac.converter.proto.getDirectCampaignProtoResponseDiffer
import ru.yandex.direct.web.entity.uac.toResponse
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService
import java.beans.PropertyEditorSupport
import java.time.Duration
import java.time.Instant
import java.time.LocalDateTime
import java.time.ZoneId
import kotlin.math.max

@RestController
@Api(tags = ["uac"])
@RequestMapping("/uac", produces = [APPLICATION_JSON_VALUE])
open class UacCampaignController(
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val authenticationSource: DirectWebAuthenticationSource,
    private val uacBannerService: UacBannerService,
    private val webServiceHolder: UacCampaignWebServiceHolder,
    private val serviceHolder: UacCampaignServiceHolder,
    private val validationResultConversionService: ValidationResultConversionService,
    private val calltrackingSettingsService: CalltrackingSettingsService,
    private val metrikaClient: MetrikaClient,
    private val featureService: FeatureService,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val uacDbDefineService: UacDbDefineService,
    private val clientService: ClientService,
    private val uacCpmStatisticsService: UacCpmStatisticsService,
    private val assetStatService: AssetStatService,
    private val uacRejectReasonService: UacRejectReasonService,
    private val zenMetaInfoService: ZenMetaInfoService,
    private val bannerService: BannerService,
    private val campaignHistoryService: UacCampaignHistoryService,
    private val audienceSegmentsService: AudienceSegmentsService,
    private val grutApiService: GrutApiService,
    @Value("\${object_api.retries}") private val grutRetries: Int,
) {
    private val recalculateAggregatedStatusesProperty =
        ppcPropertiesSupport.get(
            PpcPropertyNames.RECALCULATE_AGGREGATED_STATUSES_ON_UPDATE_STATUS_UC,
            Duration.ofMinutes(5)
        )
    private val startSearchStatsFromDateForUac =
        ppcPropertiesSupport.get(PpcPropertyNames.START_SEARCH_STATS_FROM_DATE_FOR_UAC, Duration.ofMinutes(5))
    private val changeRejectedAssetToCreatedWithSecondsShift =
        ppcPropertiesSupport.get(CHANGE_REJECTED_ASSET_TO_CREATED_WITH_SECONDS_SHIFT, Duration.ofMinutes(5))
    private val getCampaignProtoResponseDiffPercent =
        ppcPropertiesSupport.get(UAC_GET_CAMPAIGN_PROTO_RESPONSE_DIFF_PERCENT, Duration.ofMinutes(5))

    companion object {
        private const val PARAMETER_SORT_BY = "sortby"
        private const val PARAMETER_ORDER_BY = "orderby"
        private const val PARAMETER_MEDIA_TYPE = "media_type"
        private const val PARAMETER_PROTO_FORMAT = "proto_format"
        private val logger = LoggerFactory.getLogger(UacCampaignController::class.java)
        private const val DEFAULT_SECONDS_SHIFT_FOR_CHANGING_REJECTED_ASSET_TO_CREATED = 60
    }

    @ApiOperation(
        value = "campaigns",
        httpMethod = "POST",
        nickname = "campaigns",
    )
    @PreAuthorizeWrite
    @PostMapping(value = ["campaigns"])
    open fun createCampaign(
        @RequestBody request: CreateCampaignRequest,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?,
    ): ResponseEntity<Any> {
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser
        val clientId = subjectUser.clientId
        val operatorUid = operator.uid
        val cloneFromCampaignId = request.cloneFromCampaignId

        val useGrut = if (cloneFromCampaignId != null) {
            uacDbDefineService.useGrutForDirectCampaignId(cloneFromCampaignId)
        } else {
            uacDbDefineService.useGrutForNewCampaign(clientId, request.advType, nvl(request.isEcom, false))
        }
        serviceHolder.getUacCampaignService(useGrut).getOrCreateClient(operator = operator, subjectUser = subjectUser)
        val service = webServiceHolder.getUacCampaignAddService(useGrut)

        val goals = try {
            collectGoals(clientId, request.goals, request.goalCreateRequest)
        } catch (e: RuntimeException) {
            logger.error("Failed to create goal by request ${request.goalCreateRequest}", e)
            return metrikaGoalCreationError()
        }
        val calltrackingSettingsResult = saveCalltrackingSettings(
            clientId,
            operatorUid,
            request.counters,
            goals,
            request.href,
            request.calltrackingPhones
        )
        if (calltrackingSettingsResult?.isSuccessful == false) {
            return buildCalltrackingOnSiteError(calltrackingSettingsResult)
        }
        val calltrackingSettingsId = calltrackingSettingsResult?.result

        val zenPublisherId = if (request.advType == AdvType.TEXT) {
            zenMetaInfoService.getPublisherItemIdByUrl(request.href)
        } else {
            null
        }
        val internalRequest = request.toInternal(
            calltrackingSettingsId = calltrackingSettingsId,
            goals = goals,
            zenPublisherId = zenPublisherId
        )

        val result = service.addUacCampaign(
            operator = operator,
            subjectUser = subjectUser,
            internalRequest,
        )

        if (!result.isSuccessful) {
            val errors = result.validationResult.flattenErrors().joinToString("; ")
            logger.error("Cannot create campaign: {}", errors)
            return ResponseEntity(
                validationResultConversionService.buildValidationResponse(
                    result.validationResult,
                    UAC_PATH_NODE_CONVERTER_PROVIDER
                ),
                HttpStatus.BAD_REQUEST,
            )
        }

        return ResponseEntity(result.result!!.toResponse(), HttpStatus.CREATED)
    }

    @ApiOperation(
        value = "campaignGet",
        httpMethod = "GET",
        nickname = "campaignGet",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["campaign/direct/{id}"])
    open fun getDirectCampaign(
        @PathVariable id: Long,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?,
        @RequestParam(value = PARAMETER_PROTO_FORMAT, required = false) protoFormat: Boolean?,
        @RequestHeader headers: HttpHeaders,
    ): ResponseEntity<Any> {
        val subjectUser = authenticationSource.authentication.subjectUser
        val clientId = subjectUser.clientId
        val operator = authenticationSource.authentication.operator

        val useGrut = uacDbDefineService.useGrutForDirectCampaignId(id)

        val uacCampaignService = serviceHolder.getUacCampaignService(useGrut)
        val campaignId = uacCampaignService.getCampaignIdByDirectCampaignId(id) ?: return notFoundResponse();

        val useProtoFormat = protoFormat ?: false
        val protoResponseCreator = {
            getCampaignImpl(
                useGrut = useGrut,
                campaignId = campaignId,
                directCampaignId = id,
                operator = operator,
                subjectUser = subjectUser,
                clientId = clientId,
                useProtoFormat = true,
            )
        }

        if (useProtoFormat) {
            return protoResponseCreator()
        }

        val differ = getDirectCampaignProtoResponseDiffer { getCampaignProtoResponseDiffPercent.getOrDefault(0) }

        return differ.process(
            oldResponseCreator = {
                getCampaignImpl(
                    useGrut = useGrut,
                    campaignId = campaignId,
                    directCampaignId = id,
                    operator = operator,
                    subjectUser = subjectUser,
                    clientId = clientId,
                    useProtoFormat = false,
                )
            },
            protoResponseCreator = protoResponseCreator,
        ).response
    }

    @ApiOperation(
        value = "campaignHistoryGet",
        httpMethod = "GET",
        nickname = "campaignHistoryGet",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["campaign/{id}/history"])
    open fun getCampaignHistory(
        @PathVariable id: Long,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?,
        @RequestParam(value = "from", required = false) from: Long?,
        @RequestParam(value = "to") to: Long,
    ): ResponseEntity<Any> {
        val useGrut = uacDbDefineService.useGrutForDirectCampaignId(id)

        if (useGrut) {
            val historyResults = campaignHistoryService.getHistory(id, from, to)

            return if (historyResults.isSuccessful) {
                ResponseEntity(
                    historyResults.result.toResponse(),
                    null,
                    HttpStatus.OK
                )
            } else {
                ResponseEntity(
                    historyResults.validationResult,
                    null,
                    HttpStatus.BAD_REQUEST
                )
            }
        }

        logger.error("Failed to obtain history for campaign $id ($from - $to), campaign not in GrUT")
        return ResponseEntity(
            ValidationResult.failed(id, CommonDefects.objectNotFound()),
            null,
            HttpStatus.BAD_REQUEST
        )
    }

    @ApiOperation(
        value = "contentStatistics",
        httpMethod = "GET",
        nickname = "contentStatistics",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["campaign/{id}/content_statistics"])
    open fun getCampaignContentStatistics(
        @PathVariable id: Long,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?,
        @RequestParam(value = PARAMETER_SORT_BY, required = false) sortBy: String?,
        @RequestParam(value = PARAMETER_ORDER_BY, required = false) orderBy: String?,
        @RequestParam(value = PARAMETER_MEDIA_TYPE, required = false) mediaType: MediaType?,
        @RequestParam(
            value = "from",
        ) creativesStatisticsFrom: Long,
        @RequestParam(
            value = "to",
        ) creativesStatisticsTo: Long,
    ): ResponseEntity<Any> {
        val subjectUser = authenticationSource.authentication.subjectUser
        val operator = authenticationSource.authentication.operator

        val useGrut = uacDbDefineService.useGrutForDirectCampaignId(id)

        return getCampaignContentStatisticsImpl(
            useGrut,
            id,
            operator,
            subjectUser.clientId,
            subjectUser.uid,
            sortBy,
            orderBy,
            mediaType,
            creativesStatisticsFrom,
            creativesStatisticsTo,
        )
    }

    private fun getCampaignContentStatisticsImpl(
        useGrut: Boolean,
        id: Long,
        operator: User,
        clientId: ClientId,
        clientUid: Long,
        sortBy: String?,
        orderBy: String?,
        mediaType: MediaType?,
        creativesStatisticsFrom: Long,
        creativesStatisticsTo: Long
    ): ResponseEntity<Any> {
        val uacCampaignService = serviceHolder.getUacCampaignService(useGrut)
        val uacCampaignId = uacCampaignService.getCampaignIdByDirectCampaignId(id) ?: return notFoundResponse()
        val uacCampaign = uacCampaignService.getCampaignById(uacCampaignId) ?: return notFoundResponse()

        if (!uacCampaignService.checkVisibleCampaign(operator.uid, clientId, id)) {
            return notOwnerResponse()
        }

        val uacContentService = serviceHolder.getUacContentService(useGrut)
        val uacCampaignContents = uacContentService.getCampaignContents(uacCampaign)
            .filter { mediaType == null || it.type == mediaType }

        var searchFromTime: Long
        val existedCampaignContentAtPeriod: List<UacYdbCampaignContent>
        if (uacCampaign.advType == AdvType.CPM_BANNER || uacCampaign.advType == AdvType.MOBILE_CONTENT) {
            searchFromTime = creativesStatisticsFrom
            existedCampaignContentAtPeriod = uacCampaignContents
        } else {
            val campaignCreateAt = UacYdbUtils.toEpochSecond(uacCampaign.createdAt)
            searchFromTime = max(campaignCreateAt, creativesStatisticsFrom)

            // Начинаем смотреть статистику не раньше даты отправки всех хешей в БК,
            // т.к. до этой даты хеши отправлялись не полным составом
            searchFromTime = max(searchFromTime, startSearchStatsFromDateForUac.getOrDefault(0L))

            // Отбираем только существовавшие ассеты в выбранный промежуток времени
            existedCampaignContentAtPeriod = assetStatService.getExistedCampaignContentAtPeriod(
                uacCampaignContents,
                searchFromTime,
                creativesStatisticsTo,
            )
        }

        val mediaCampaignContents = existedCampaignContentAtPeriod.filter { it.contentId != null }
        val uacContentById = uacContentService.getDbContents(mediaCampaignContents.map { it.contentId!! })
            .associateBy { it.id }

        val creativesStatistics: Map<String, CreativeStatistics>
        val campaignContentIdToEfficiencyWithExistenceAndStat: Map<String, ContentEfficiencyWithExistenceAndStat>
        if (uacCampaign.advType == AdvType.CPM_BANNER) {
            creativesStatistics = uacCpmStatisticsService.getCreativesStatisticsFromMoc(
                mediaCampaignContents.associate { it.id to uacContentById[it.contentId]!! },
                clientUid,
                operator.uid,
                id,
                Instant.ofEpochSecond(searchFromTime).atZone(ZoneId.systemDefault()).toLocalDate(),
                Instant.ofEpochSecond(creativesStatisticsTo).atZone(ZoneId.systemDefault()).toLocalDate()
            )
            campaignContentIdToEfficiencyWithExistenceAndStat = emptyMap()
        } else {
            campaignContentIdToEfficiencyWithExistenceAndStat = assetStatService.getAssetsEfficiencyAndExistence(
                existedCampaignContentAtPeriod,
                id,
                searchFromTime,
                creativesStatisticsTo,
            )
            logger.info("Statistical data: $campaignContentIdToEfficiencyWithExistenceAndStat")

            creativesStatistics = campaignContentIdToEfficiencyWithExistenceAndStat
                .mapValues { it.value.assetStat.toCreativeStatistics() }
        }

        val language = getCurrentLanguage()
        val campaignContentIdToRejectReasons =
            uacRejectReasonService.getRejectReasons(existedCampaignContentAtPeriod, language)

        val changeRejectedAssetToCreatedBySecondsShift = changeRejectedAssetToCreatedWithSecondsShift
            .getOrDefault(DEFAULT_SECONDS_SHIFT_FOR_CHANGING_REJECTED_ASSET_TO_CREATED)

        val bidByCreativeId = bannerService
            .getCreativeIdToBidsByCampaignId(clientId, id)
            .entries
            .associate { (k, v) -> k to v[0] }
        var uacContentStatistics = existedCampaignContentAtPeriod
            .map {
                val content = uacContentById[it.contentId]
                val efficiency = campaignContentIdToEfficiencyWithExistenceAndStat[it.id]?.contentEfficiency
                val existence = campaignContentIdToEfficiencyWithExistenceAndStat[it.id]?.existence
                val rejectReasons = campaignContentIdToRejectReasons[it.id]
                val creativeId = when (val creativeIdJsonValue = content?.meta?.get("creative_id")) {
                    is Int -> creativeIdJsonValue.toLong()
                    is Long -> creativeIdJsonValue
                    else -> {
                        logger.warn("wrong creativeId for banner with contentId: {}", it.contentId)
                        null
                    }
                }
                toContentStatistics(
                    it,
                    uacContentService.fillContent(content),
                    efficiency,
                    partialExistence = existence,
                    creativeStatistics = creativesStatistics[it.id],
                    rejectReasons = rejectReasons,
                    changeRejectedAssetToCreatedBySecondsShift,
                    bid = bidByCreativeId[creativeId]?.toString(),
                )
            }

        val contentStatisticsSortBy = ContentStatisticsSortBy.fromId(sortBy) ?: ContentStatisticsSortBy.ID
        val contentStatisticsOrderBy = ContentStatisticsOrderBy.fromId(orderBy) ?: ContentStatisticsOrderBy.ASC

        uacContentStatistics = uacContentStatistics.sortedWith(compareBy {
            when (contentStatisticsSortBy) {
                ContentStatisticsSortBy.EFFICIENCY -> it.efficiency?.ordinal.toString()
                ContentStatisticsSortBy.CONTENT_ID -> it.content?.id
                ContentStatisticsSortBy.SHOWS -> it.creativeStatistics?.shows
                ContentStatisticsSortBy.CLICKS -> it.creativeStatistics?.clicks
                ContentStatisticsSortBy.CONVERSIONS -> it.creativeStatistics?.conversions
                ContentStatisticsSortBy.CTR -> it.creativeStatistics?.ctr
                ContentStatisticsSortBy.CR -> it.creativeStatistics?.cr
                ContentStatisticsSortBy.CPA -> it.creativeStatistics?.cpa
                ContentStatisticsSortBy.ID -> it.id
            }
        })

        if (ContentStatisticsOrderBy.DESC == contentStatisticsOrderBy) {
            uacContentStatistics = uacContentStatistics.reversed()
        }

        return ResponseEntity(
            toResponse(uacContentStatistics, UacNavigationInfo(contentStatisticsSortBy, contentStatisticsOrderBy)),
            null,
            HttpStatus.OK
        )
    }

    @ApiOperation(
        value = "campaignGet",
        httpMethod = "GET",
        nickname = "campaignGet",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["campaign/{id}"])
    open fun getCampaignById(
        @PathVariable id: String,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?
    ): ResponseEntity<Any> {
        if (!isValidId(id)) {
            logger.error("Invalid campaign id $id")
            return notFoundResponse()
        }
        val subjectUser = authenticationSource.authentication.subjectUser
        val clientId = subjectUser.clientId
        val operator = authenticationSource.authentication.operator
        val useGrut = uacDbDefineService.useGrut(id)

        val uacCampaignService = serviceHolder.getUacCampaignService(useGrut)
        val directCampaignId = uacCampaignService.getDirectCampaignIdById(id) ?: return notFoundResponse()
        return getCampaignImpl(useGrut, id, directCampaignId, operator, subjectUser, clientId, useProtoFormat = false)
    }

    private fun getCampaignImpl(
        useGrut: Boolean,
        campaignId: String,
        directCampaignId: Long,
        operator: User,
        subjectUser: User,
        clientId: ClientId,
        useProtoFormat: Boolean,
    ): ResponseEntity<Any> {
        val uacCampaignService = serviceHolder.getUacCampaignService(useGrut)
        val uacCampaignWebService = webServiceHolder.getUacCampaignWebService(useGrut)

        if (useProtoFormat) {
            val campaignProto = uacCampaignService.getCampaignProtoByDirectCampaignId(directCampaignId)
                ?: return notFoundResponse()

            if (!uacCampaignService.checkVisibleCampaign(operator.uid, clientId, directCampaignId)) {
                return notOwnerResponse()
            }
            val responseProto = uacCampaignWebService.fillCampaignProto(
                operator = operator, subjectUser = subjectUser,
                campaign = campaignProto,
                clientId = clientId,
            )
            return ResponseEntity(responseProto.toResponse(), HttpStatus.OK)
        } else {
            val campaign = uacCampaignService.getCampaignByDirectCampaignId(directCampaignId)
                ?: return notFoundResponse()

            if (!uacCampaignService.checkVisibleCampaign(operator.uid, clientId, directCampaignId)) {
                return notOwnerResponse()
            }

            val bid = uacCampaignWebService.getMinBannerIdForCampaign(campaignId)

            val campaignStatuses = uacCampaignWebService.recalcStatuses(
                clientId = clientId,
                directCampaignId = directCampaignId,
                isDraft = campaign.isDraft,
                bid = bid
            ) ?: return notFoundResponse()

            val updatedAndFilledCampaign =
                uacCampaignWebService.fillCampaign(
                    operator = operator, subjectUser = subjectUser,
                    directCampaignId = directCampaignId,
                    campaign = campaign,
                    campaignStatuses = campaignStatuses,
                    bid = bid
                )

            return ResponseEntity(updatedAndFilledCampaign.toResponse(), HttpStatus.OK)
        }
    }

    @ApiOperation(
        value = "campaignUpdate",
        httpMethod = "PATCH",
        nickname = "campaignUpdate",
    )
    @PreAuthorizeWrite
    @PatchMapping(value = ["campaign/{id}"])
    open fun updateCampaign(
        @RequestBody request: PatchCampaignRequest,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?,
        @PathVariable id: String,
    ): ResponseEntity<Any> {
        if (!isValidId(id)) {
            logger.error("Invalid campaign id $id")
            return notFoundResponse()
        }
        val useGrut = uacDbDefineService.useGrut(id)
        return grutTransactionProvider.runInRetryableTransactionIfNeeded(useGrut, grutRetries) {
            serviceHolder.getUacCampaignService(useGrut)
                .getCampaignById(id)?.let { uacYdbCampaign ->
                    updateCampaign(request, ulogin, uacYdbCampaign, useGrut)
                } ?: notFoundResponse()
        }
    }

    @ApiOperation(
        value = "campaignUpdate",
        httpMethod = "PATCH",
        nickname = "campaignUpdate",
    )
    @PreAuthorizeWrite
    @PatchMapping(value = ["campaign/direct/{id}"])
    open fun updateCampaignDirect(
        @RequestBody request: PatchCampaignRequest,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?,
        @PathVariable id: Long,
    ): ResponseEntity<Any> {
        val useGrut = uacDbDefineService.useGrutForDirectCampaignId(id)
        return grutTransactionProvider.runInTransactionIfNeeded(useGrut) {
            val campaignService = serviceHolder.getUacCampaignService(useGrut)
            campaignService.getCampaignIdByDirectCampaignId(id)?.let { campaignId ->
                campaignService.getCampaignById(campaignId)
            }?.let { campaign ->
                updateCampaign(request, ulogin, campaign, useGrut)
            } ?: notFoundResponse()
        }
    }

    private fun updateCampaign(
        request: PatchCampaignRequest,
        ulogin: String?,
        uacYdbCampaign: UacYdbCampaign,
        useGrut: Boolean,
    ): ResponseEntity<Any> {
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser

        val uacCampaignService = serviceHolder.getUacCampaignService(useGrut)

        var directCampaignId = uacCampaignService.getDirectCampaignIdById(uacYdbCampaign.id)
            ?: return notFoundResponse()

        val clientId = subjectUser.clientId
        val multipleAdsInUc = featureService.isEnabledForClientId(clientId, FeatureName.UC_MULTIPLE_ADS_ENABLED)
        val updateMinusKeywords =
            featureService.isEnabledForClientId(clientId, FeatureName.UAC_ENABLE_MINUS_KEYWORDS_TGO)

        val operatorUid = operator.uid
        val isWritable = uacCampaignService.checkWritableCampaign(
            operatorUid,
            clientId,
            directCampaignId
        )
        if (!isWritable) {
            return notOwnerResponse()
        }

        if (!uacYdbCampaign.isDraft && request.appId != null) {
            return actionNotAllowedResponse()
        }
        val campaignStatuses = uacCampaignService.getCampaignStatuses(clientId, directCampaignId, uacYdbCampaign)
            ?: return notFoundResponse()
        if (campaignStatuses.status == Status.ARCHIVED) {
            return campaignIsArchivedResponse()
        }

        val goals = try {
            collectGoals(clientId, request.goals, request.goalCreateRequest)
        } catch (e: RuntimeException) {
            logger.error("Failed to create goal by request ${request.goalCreateRequest}", e)
            return metrikaGoalCreationError()
        }
        val calltrackingSettingsResult = saveCalltrackingSettings(
            clientId,
            operatorUid,
            request.counters,
            goals,
            request.href,
            request.calltrackingPhones
        )
        if (calltrackingSettingsResult?.isSuccessful == false) {
            return buildCalltrackingOnSiteError(calltrackingSettingsResult)
        }
        val calltrackingSettingsId = calltrackingSettingsResult?.result
        val zenPublisherId = request.href?.let { zenMetaInfoService.getPublisherItemIdByUrl(it) }
        val internalRequest = request.toInternal(
            calltrackingSettingsId = calltrackingSettingsId, goals = goals, zenPublisherId = zenPublisherId
        )
        // правим возможную неконсистентность: у ecom-кампаний не должно быть управления от Яндекса,
        // силами фронта это полностью не решается.
        fixRecommendationsManagement(internalRequest, uacYdbCampaign.isEcom)
        val updateCampaignResult = webServiceHolder.getUacCampaignUpdateService(useGrut)
            .updateCampaign(
                uacYdbCampaign,
                directCampaignId,
                internalRequest,
                operator,
                subjectUser,
                multipleAdsInUc,
                updateMinusKeywords
            )

        if (!updateCampaignResult.isSuccessful) {
            val errors = updateCampaignResult.validationResult.flattenErrors().joinToString("; ")
            logger.error("Cannot update campaign {}: {}", uacYdbCampaign.id, errors)
            return ResponseEntity(
                validationResultConversionService.buildValidationResponse(
                    updateCampaignResult.validationResult,
                    UAC_PATH_NODE_CONVERTER_PROVIDER
                ),
                HttpStatus.BAD_REQUEST,
            )
        }

        val updatedYdbUacCampaign = updateCampaignResult.result!!

        directCampaignId = uacCampaignService.getDirectCampaignIdById(updatedYdbUacCampaign.id)!!
        val updatedUacCampaign = webServiceHolder.getUacCampaignWebService(useGrut).fillCampaign(
            operator = operator, subjectUser = subjectUser,
            updatedYdbUacCampaign, directCampaignId, campaignStatuses
        )

        return ResponseEntity(updatedUacCampaign.toResponse(), HttpStatus.OK)
    }

    @ApiOperation(
        value = "campaignDelete",
        httpMethod = "DELETE",
        nickname = "campaignDelete",
    )
    @PreAuthorizeWrite
    @DeleteMapping(value = ["campaign/{id}"])
    open fun deleteCampaign(
        @PathVariable id: String,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?
    ): ResponseEntity<Any> {
        if (!isValidId(id)) {
            logger.error("Invalid campaign id $id")
            return notFoundResponse()
        }

        val subjectUser = authenticationSource.authentication.subjectUser
        val service = serviceHolder.getUacCampaignService(id)

        val directCampaignId = service.getDirectCampaignIdById(id) ?: return notFoundResponse()

        return deleteCampaign(id, directCampaignId, subjectUser, service)
    }

    @ApiOperation(
        value = "campaignDelete",
        httpMethod = "DELETE",
        nickname = "campaignDelete",
    )
    @PreAuthorizeWrite
    @DeleteMapping(value = ["campaign/direct/{id}"])
    open fun deleteCampaignDirect(
        @PathVariable id: String,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?
    ): ResponseEntity<Any> {
        if (!isValidId(id)) {
            logger.error("Invalid campaign id $id")
            return notFoundResponse()
        }

        val subjectUser = authenticationSource.authentication.subjectUser
        val service = serviceHolder.getUacCampaignServiceForDirectCampaignId(id.toLong())
        val campaignId = service.getCampaignIdByDirectCampaignId(id.toLong()) ?: return notFoundResponse()
        return deleteCampaign(campaignId, id.toLong(), subjectUser, service)
    }

    private fun deleteCampaign(
        campaignId: String,
        directCampaignId: Long,
        subjectUser: User,
        service: BaseUacCampaignService,
    ): ResponseEntity<Any> {
        val operator = authenticationSource.authentication.operator
        val clientId = subjectUser.clientId

        val campaign = service.getCampaignById(campaignId) ?: return notFoundResponse()
        if (!campaign.isDraft) {
            return actionNotAllowedResponse()
        }

        val result = service.deleteCampaign(campaignId, directCampaignId, operator.uid, clientId)
        return if (!result.isSuccessful) {
            ResponseEntity(
                validationResultConversionService.buildValidationResponse(result),
                HttpStatus.BAD_REQUEST
            )
        } else ResponseEntity(HttpStatus.NO_CONTENT)
    }

    @ApiOperation(
        value = "campaignStatus",
        httpMethod = "POST",
        nickname = "campaignStatus",
    )
    @PreAuthorizeWrite
    @PostMapping(value = ["campaign/{id}/status"], produces = [APPLICATION_JSON_VALUE])
    open fun postCampaignStatus(
        @RequestBody campaignStatusesRequest: CampaignStatusesRequest,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?,
        @PathVariable id: String
    ): ResponseEntity<Any> {
        if (!isValidId(id)) {
            logger.error("Invalid campaign id $id")
            return notFoundResponse()
        }
        val useGrut = uacDbDefineService.useGrut(id)
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser
        val dbAccountId = serviceHolder.getUacCampaignService(useGrut)
            .getOrCreateClient(subjectUser = subjectUser, operator = operator)
        return grutTransactionProvider.runInRetryableTransactionIfNeeded(useGrut, grutRetries) f@{
            val uacCampaignService = serviceHolder.getUacCampaignService(useGrut)
            val uacYdbCampaign = uacCampaignService.getCampaignById(id) ?: return@f notFoundResponse()
            val campaignStatuses = uacCampaignService.getCampaignStatuses(
                subjectUser.clientId,
                uacCampaignService.getDirectCampaignIdById(id)!!,
                uacYdbCampaign
            ) ?: return@f notFoundResponse()
            postCampStatus(campaignStatusesRequest, uacYdbCampaign, dbAccountId, useGrut, campaignStatuses)
        }
    }

    @ApiOperation(
        value = "campaignStatus",
        httpMethod = "POST",
        nickname = "campaignStatus",
    )
    @PreAuthorizeWrite
    @PostMapping(value = ["campaign/direct/{id}/status"], produces = [APPLICATION_JSON_VALUE])
    open fun postDirectCampaignStatus(
        @RequestBody campaignStatusesRequest: CampaignStatusesRequest,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?,
        @PathVariable id: String
    ): ResponseEntity<Any> {
        if (!isValidId(id)) {
            logger.error("Invalid campaign id $id")
            return notFoundResponse()
        }
        val useGrut = uacDbDefineService.useGrutForDirectCampaignId(id.toLong())
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser
        val dbAccountId = serviceHolder.getUacCampaignService(useGrut)
            .getOrCreateClient(subjectUser = subjectUser, operator = operator)

        return grutTransactionProvider.runInRetryableTransactionIfNeeded(useGrut, grutRetries) f@{
            val campaignService = serviceHolder.getUacCampaignService(useGrut)
            val uacYdbCampaign = campaignService.getCampaignIdByDirectCampaignId(id.toLong())?.let { campaignId ->
                campaignService.getCampaignById(campaignId)
            } ?: return@f notFoundResponse()
            val campaignStatuses =
                campaignService.getCampaignStatuses(subjectUser.clientId, id.toIdLong(), uacYdbCampaign)
                    ?: return@f notFoundResponse()
            postCampStatus(campaignStatusesRequest, uacYdbCampaign, dbAccountId, useGrut, campaignStatuses)
        }
    }

    private fun postCampStatus(
        campaignStatusesRequest: CampaignStatusesRequest,
        ydbCampaign: UacYdbCampaign,
        accountId: String,
        useGrut: Boolean,
        campaignStatuses: CampaignStatuses,
    ): ResponseEntity<Any> {

        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser

        val campaignWebService = webServiceHolder.getUacCampaignWebService(useGrut)
        val campaignService = serviceHolder.getUacCampaignService(useGrut)
        val campaignUpdateService = serviceHolder.getUacCampaignUpdateService(useGrut)
        val webCampaignUpdateService = webServiceHolder.getUacCampaignUpdateService(useGrut)

        if (accountId != ydbCampaign.account) {
            return notOwnerResponse()
        }
        if (campaignStatuses.status == Status.ARCHIVED) {
            return campaignIsArchivedResponse()
        }

        val newTargetStatus = campaignStatusesRequest.targetStatus

        val directCampaignStatus = campaignService.getDirectCampaignStatus(ydbCampaign)
        val directCampaignId = campaignService.getDirectCampaignIdById(ydbCampaign.id)

        val directCampaignStatusShow = webCampaignUpdateService.getCampaignStatusShow(directCampaignId!!)

        // Возвращаем ошибку только если новый статус = статусу кампании и заявки, иначе пересчитываем
        if (newTargetStatus == TargetStatus.STARTED
            && ydbCampaign.targetStatus == TargetStatus.STARTED
            && directCampaignStatusShow == true
        ) {
            return campaignAlreadyStartedResponse()
        }

        if (newTargetStatus == TargetStatus.STARTED
            && ydbCampaign.advType == AdvType.CPM_BANNER
            && featureService.isEnabledForClientId(subjectUser.clientId,
                FeatureName.IS_CPM_BANNER_CAMPAIGN_DISABLED
            )) {
            return cantStartCpmBannerCampaign()
        }
        if (newTargetStatus == TargetStatus.STOPPED
            && ydbCampaign.targetStatus == TargetStatus.STOPPED
            && directCampaignStatusShow == false
        ) {
            return campaignAlreadyStoppedResponse()
        }

        val targetShowStatus = ydbCampaign.targetStatus == TargetStatus.STARTED
        if (targetShowStatus != directCampaignStatusShow) {
            logger.warn("Different statuses in grut '$targetShowStatus' and campaign '$directCampaignStatusShow'" +
                " for campaign id $directCampaignId")
        }

        var newCampaignStatus: Status = campaignStatuses.status

        val uacCampaignModelChanges: KtModelChanges<String, UacYdbCampaign> = KtModelChanges(ydbCampaign.id)

        if (newTargetStatus == TargetStatus.STARTED) {
            if (directCampaignStatus != null && directCampaignStatus != DirectCampaignStatus.DRAFT) {
                if (!campaignService.updateDirectCampaignStatusShow(
                        operator,
                        subjectUser.clientId,
                        directCampaignId,
                        ydbCampaign,
                        true
                    )
                ) {
                    logger.error("cant update status show")
                    return notAllowedResponse()
                }

                val statuses = campaignWebService.recalcStatuses(
                    clientId = subjectUser.clientId,
                    directCampaignId = directCampaignId,
                    isDraft = ydbCampaign.isDraft,
                    withRecalculatingAggrStatuses = recalculateAggregatedStatusesProperty.getOrDefault(false),
                )
                newCampaignStatus = statuses!!.status
            } else {
                val validationResult = webServiceHolder.getUacContentWebService(useGrut)
                    .checkCampaignIsCompleteForUpdateStatus(ydbCampaign)

                if (validationResult != null) {
                    return validationResult
                }
                if (!webCampaignUpdateService.setDirectCampaignStatusToCreated(
                        ydbCampaign,
                        operator,
                        subjectUser,
                        uacCampaignModelChanges
                    )
                ) {
                    return errorCreatingDirectCampaignResponse()
                }
                uacCampaignModelChanges.process(UacYdbCampaign::briefSynced, false)
                uacBannerService.updateAdsDeferred(subjectUser.clientId, operator.uid, ydbCampaign.id)
                newCampaignStatus = Status.MODERATING
            }
            if (featureService.isEnabledForClientId(
                    subjectUser.clientId,
                    FeatureName.CHECK_AUDIENCE_SEGMENTS_DEFERRED
                ) && audienceSegmentsService.hasProcessedSegments(ydbCampaign.retargetingCondition, subjectUser.login)
            ) {
                uacCampaignModelChanges.process(UacYdbCampaign::audienceSegmentsSynchronized, false)
                audienceSegmentsService.checkAudienceSegmentsDeferred(
                    subjectUser.clientId,
                    operator.uid,
                    ydbCampaign.id
                )
            }
        } else if (newTargetStatus == TargetStatus.STOPPED) {
            if (directCampaignStatus == null || directCampaignStatus == DirectCampaignStatus.DRAFT) {
                return campaignDoesNotExistResponse()
            }
            if (!campaignService.updateDirectCampaignStatusShow(
                    operator,
                    subjectUser.clientId,
                    directCampaignId,
                    ydbCampaign,
                    false
                )
            ) {
                logger.error("cant update status show")
                return notAllowedResponse()
            }

            val statuses = campaignWebService.recalcStatuses(
                clientId = subjectUser.clientId,
                directCampaignId = directCampaignId,
                isDraft = ydbCampaign.isDraft,
                withRecalculatingAggrStatuses = recalculateAggregatedStatusesProperty.getOrDefault(false),
            )
            newCampaignStatus = statuses!!.status
        }

        uacCampaignModelChanges.process(UacYdbCampaign::targetStatus, newTargetStatus)
        if (ydbCampaign.isDraft) {
            uacCampaignModelChanges.process(UacYdbCampaign::startedAt, LocalDateTime.now())
        }
        uacCampaignModelChanges.process(UacYdbCampaign::updatedAt, LocalDateTime.now())
        campaignUpdateService.updateCampaignFromModelChanges(ydbCampaign, uacCampaignModelChanges)

        return ResponseEntity(
            CampaignStatuses(
                targetStatus = newTargetStatus,
                status = newCampaignStatus
            ),
            HttpStatus.OK
        )
    }

    /**
     * Собрать все цели, необходимые для создания/обновления кампании
     *
     * Список целей состоит из переданных в запросе `request.goals` целей и цели,
     * созданной по `request.goalCreateRequest`
     * @throws RuntimeException если не удалось создать цель
     */
    private fun collectGoals(
        clientId: ClientId,
        goals: List<UacGoal>?,
        goalRequest: GoalCreateRequest?
    ): List<UacGoal>? {
        if (goalRequest == null) {
            return goals
        }
        if (goalRequest.goalType != CreatingGoalType.CALL) {
            throw UnsupportedOperationException("Creation of goals with type ${goalRequest.goalType} is not supported")
        }
        val counterId = goalRequest.counterId
        val response = metrikaClient.turnOnCallTracking(counterId)
        val goalId = response.goal.id.toLong()

        val currency = clientService.getWorkCurrency(clientId)
        val goal = UacGoal(goalId, conversionValue = currency.ucDefaultConversionValue ?: currency.minPrice)
        return listOf(goal) + (goals ?: emptyList())
    }

    /**
     * Сохранить настройки коллтрекинга на сайте
     *
     * Идентификатор сохраненных настроек коллтрекинга на сайте или `null`,
     * если настройки с переданными параметрами нужно будет удалить с кампании, если они были
     */
    private fun saveCalltrackingSettings(
        clientId: ClientId,
        operatorUid: Long,
        counters: List<Int>?,
        goals: List<UacGoal>?,
        href: String?,
        phones: List<String>?
    ): Result<Long>? {
        if (phones.isNullOrEmpty()) {
            // Если в исходных телефонов нет, значит на этой кампании нет или должно не стать коллтрекинга
            return null
        }
        if (goals.isNullOrEmpty()) {
            // Для коллтрекинга обязательна цель Звонок в исходных данных
            logger.error("Cannot find call goals from input goals: $goals")
            return Result.broken(ValidationResult.failed(goals, CommonDefects.invalidValue()))
        }
        val callGoals = calltrackingSettingsService.getCallGoals(clientId, operatorUid, counters)
        if (callGoals.isEmpty()) {
            // Не нашли цель Звонок
            logger.error("Cannot find call goals from input, counters: $counters")
            return Result.broken(ValidationResult.failed(counters, CommonDefects.invalidValue()))
        }
        val inputGoalIds = goals.map { it.goalId }
        val intersect = inputGoalIds.intersect(callGoals.keys)
        if (intersect.isEmpty()) {
            logger.error("Cannot find call goals from input goals: $goals")
            return Result.broken(ValidationResult.failed(goals, CommonDefects.invalidValue()))
        }
        if (intersect.size > 1) {
            logger.error("More than one input goals has call type: $goals")
            return Result.broken(ValidationResult.failed(goals, CommonDefects.invalidValue()))
        }
        val counterId = callGoals[intersect.first()]?.counterId?.toLong()
        return calltrackingSettingsService.save(clientId, operatorUid, href, counterId, phones)
    }

    private fun buildCalltrackingOnSiteError(calltrackingSettingsResult: Result<Long>): ResponseEntity<Any> {
        val errors = calltrackingSettingsResult.validationResult?.flattenErrors()!!.joinToString("; ")
        logger.error("Failed to save calltracking on site: $errors")
        return ResponseEntity(
            validationResultConversionService.buildValidationResponse(
                calltrackingSettingsResult.validationResult,
                UAC_PATH_NODE_CONVERTER_PROVIDER
            ),
            HttpStatus.BAD_REQUEST
        )
    }

    // сбрасываем пару флагов ecom-кампаниям, если фронт не справился сам
    private fun fixRecommendationsManagement(internalRequest: PatchCampaignInternalRequest, ecom: Boolean?) {
        if (ecom == true) {
            internalRequest.isRecommendationsManagementEnabled = false
            internalRequest.isPriceRecommendationsManagementEnabled = false
        }
    }

    private fun metrikaGoalCreationError(): ResponseEntity<Any> {
        return ResponseEntity(
            validationResultConversionService.buildValidationResponse(
                ValidationResult.failed(null, metrikaReturnsResultWithErrors()),
                UAC_PATH_NODE_CONVERTER_PROVIDER
            ),
            HttpStatus.INTERNAL_SERVER_ERROR
        )
    }

    @InitBinder
    fun initBinder(dataBinder: WebDataBinder) {
        dataBinder.registerCustomEditor(MediaType::class.java, MediaTypeConverter())
    }

    inner class MediaTypeConverter : PropertyEditorSupport() {
        override fun setAsText(text: String?) {
            value = text?.let { MediaType.fromText(it.lowercase()) }
        }
    }
}
