package ru.yandex.direct.web.entity.uac.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiParam
import java.beans.PropertyEditorSupport
import org.slf4j.LoggerFactory
import org.springframework.http.HttpStatus
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.WebDataBinder
import org.springframework.web.bind.annotation.DeleteMapping
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.InitBinder
import org.springframework.web.bind.annotation.PathVariable
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RequestPart
import org.springframework.web.bind.annotation.RestController
import org.springframework.web.multipart.MultipartFile
import ru.yandex.direct.canvas.client.model.html5.Html5Tag
import ru.yandex.direct.common.util.HttpUtil
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.CreativeType
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder
import ru.yandex.direct.core.entity.uac.service.UacDbDefineService
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN
import ru.yandex.direct.core.entity.uac.isValidId
import ru.yandex.direct.web.entity.uac.model.CreateContentRequest
import ru.yandex.direct.web.entity.uac.notFoundResponse
import ru.yandex.direct.web.entity.uac.notOwnerResponse
import ru.yandex.direct.web.entity.uac.service.UacCampaignWebServiceHolder
import ru.yandex.direct.web.entity.uac.toResponse

@RestController
@Api(tags = ["uac"])
@RequestMapping("/uac", produces = [MediaType.APPLICATION_JSON_VALUE])
open class UacContentController(
    private val authenticationSource: DirectWebAuthenticationSource,
    private val uacDbDefineService: UacDbDefineService,
    private val serviceHolder: UacCampaignServiceHolder,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val uacCampaignWebServiceHolder: UacCampaignWebServiceHolder,
    private val uacCampaignServiceHolder: UacCampaignServiceHolder,
) {
    companion object {
        private val logger = LoggerFactory.getLogger(UacContentController::class.java)
    }

    @ApiOperation(
        value = "contentGet",
        httpMethod = "GET",
        nickname = "contentGet"
    )
    @PreAuthorizeRead
    @GetMapping(value = ["content/{id}"])
    open fun getContent(
        @PathVariable id: String,
    ): ResponseEntity<Any> {
        if (!isValidId(id)) {
            logger.error("Invalid content id $id")
            return notFoundResponse()
        }
        val useGrut = uacDbDefineService.useGrutForContentId(id)
        val contentService = serviceHolder.getUacContentService(useGrut)
        val content = contentService.getContent(id) ?: return notFoundResponse()
        return ResponseEntity(content.toResponse(), HttpStatus.OK)
    }

    @ApiOperation(
        value = "contentDelete",
        httpMethod = "DELETE",
        nickname = "contentDelete"
    )
    @PreAuthorizeRead
    @DeleteMapping(value = ["content/{id}"])
    open fun deleteContent(
        @PathVariable id: String,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?
    ): ResponseEntity<Any> {
        if (!isValidId(id)) {
            logger.error("Invalid content id $id")
            return notFoundResponse()
        }
        val client = authenticationSource.authentication.subjectUser
        val useGrut = uacDbDefineService.useGrutForContentId(id)
        val contentService = serviceHolder.getUacContentService(useGrut)

        return grutTransactionProvider.runInTransactionIfNeeded(useGrut) {
            val content = contentService.getYdbContent(id)
            if (content == null) {
                notFoundResponse()
            } else if (!contentService.checkVisibleContent(client.clientId, content)) {
                notOwnerResponse()
            } else {
                contentService.deleteContent(id)
                ResponseEntity(HttpStatus.NO_CONTENT)
            }
        }
    }

    /**
     * NB! в это ручке не создается транзакция GrUT, так как в ручке есть походы в canvas и avatars, и пишущий запрос всего один
     * При добавлении еще пишущего запроса, нужно сгруппировать их в одну транзакцию(НО без походов по сети в другие сервисы)
     */
    @ApiOperation(
        value = "contentCreateByFile",
        httpMethod = "POST",
        nickname = "contentCreateByFile"
    )
    @PreAuthorizeWrite
    @PostMapping(
        value = ["content", "content/by/file"],
        consumes = [MediaType.MULTIPART_FORM_DATA_VALUE],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    open fun createContentByFile(
        @ApiParam(value = "adv_type", required = false) @RequestParam(value = "adv_type") advType: AdvType?,
        @ApiParam(value = "is_ecom", required = false) @RequestParam(value = "is_ecom") isEcom: Boolean?,
        @ApiParam(value = "creative_type", required = false) @RequestParam(value = "creative_type") creativeType: CreativeType?,
        @ApiParam(value = "campaign_id", required = false) @RequestParam(value = "campaign_id") campaignId: String?,
        @ApiParam(value = "html5_tag", required = false) @RequestParam(value = "html5_tag") html5Tag: Html5Tag?,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?,
        @ApiParam(name = "upload", value = "upload", required = true) @RequestPart(
            value = "upload",
            required = true
        ) multipartFile: MultipartFile,
    ): ResponseEntity<Any> {
        return createContent(advType, isEcom, creativeType, multipartFile, null, campaignId, html5Tag)
    }

    /**
     * NB! в это ручке не создается транзакция GrUT, так как в ручке есть походы в canvas и avatars, и пишущий запрос всего один
     * При добавлении еще пишущего запроса, нужно сгруппировать их в одну транзакцию(НО без походов по сети в другие сервисы)
     */
    @ApiOperation(
        value = "contentCreateByUrl",
        httpMethod = "POST",
        nickname = "contentCreateByUrl"
    )
    @PreAuthorizeWrite
    @PostMapping(
        value = ["content", "content/by/url"],
        consumes = [MediaType.APPLICATION_JSON_UTF8_VALUE],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    open fun createContentByUrl(
        @ApiParam(value = "adv_type", required = false) @RequestParam(value = "adv_type") advType: AdvType?,
        @ApiParam(value = "is_ecom", required = false) @RequestParam(value = "is_ecom") isEcom: Boolean?,
        @ApiParam(value = "creative_type", required = false) @RequestParam(value = "creative_type") creativeType: CreativeType?,
        @ApiParam(value = "campaign_id", required = false) @RequestParam(value = "campaign_id") campaignId: String?,
        @RequestParam(value = PARAMETER_ULOGIN) ulogin: String?,
        @RequestBody createContentRequest: CreateContentRequest,
    ): ResponseEntity<Any> {
        return createContent(advType, isEcom, creativeType, null, createContentRequest, campaignId, null)
    }

    @ApiOperation(
        value = "contentGetByHash",
        httpMethod = "GET",
        nickname = "contentGetByHash"
    )
    @PreAuthorizeRead
    @GetMapping(value = ["content/hash/{hash}"])
    open fun getContentByHash(
        @PathVariable hash: String,
    ): ResponseEntity<Any> {
        val useGrut = uacDbDefineService.useGrutForContentHash(hash)
        val contentService = serviceHolder.getUacContentService(useGrut)
        val content = contentService.getContentByHash(hash) ?: return notFoundResponse()
        return ResponseEntity(content.toResponse(), HttpStatus.OK)
    }

    /**
     * NB! в этом методе не создается транзакция GrUT, так как в ручке есть походы в canvas и avatars, и пишущий запрос всего один
     * При добавлении еще пишущего запроса, нужно сгруппировать их в одну транзакцию(НО без походов по сети в другие сервисы)
     */
    private fun createContent(
        advType: AdvType?,
        isEcom: Boolean?,
        creativeType: CreativeType?,
        multipartFile: MultipartFile?,
        createContentRequest: CreateContentRequest?,
        campaignId: String?,
        html5Tag: Html5Tag?,
    ): ResponseEntity<Any> {
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser

        val useGrut = when {
            campaignId != null -> {
                uacDbDefineService.useGrutForDirectCampaignId(campaignId.toIdLong())
            }
            advType != null -> {
                uacDbDefineService.useGrutForNewCampaign(subjectUser.clientId, advType, isEcom ?: false)
            }
            else -> {
                // на данный момент параметр advType не передается с фронта, как начнет передаваться, убрать эту ветку
                uacDbDefineService.useGrutForNewCampaign(subjectUser.clientId)
            }
        }
        val accountId = uacCampaignServiceHolder.getUacCampaignService(useGrut).getOrCreateClient(operator, subjectUser)
        return uacCampaignWebServiceHolder.getUacContentWebService(useGrut)
            .createContent(
                operator,
                subjectUser,
                creativeType ?: CreativeType.RMP,
                multipartFile,
                createContentRequest,
                accountId,
                HttpUtil.getCurrentLocale().orElse(null),
                html5Tag ?: Html5Tag.PLAYABLE
            )
    }

    @InitBinder
    fun initBinder(dataBinder: WebDataBinder) {
        dataBinder.registerCustomEditor(CreativeType::class.java, CreativeTypeConverter())
        dataBinder.registerCustomEditor(AdvType::class.java, AdvTypeConverter())
    }

    inner class CreativeTypeConverter : PropertyEditorSupport() {
        override fun setAsText(text: String?) {
            value = text?.let { CreativeType.fromId(text.lowercase()) }
        }
    }

    inner class AdvTypeConverter : PropertyEditorSupport() {
        override fun setAsText(text: String?) {
            value = text?.let { AdvType.fromName(text.uppercase()) }
        }
    }
}
