package ru.yandex.direct.web.entity.uac.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiParam
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RestController
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.i18n.Language
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource
import ru.yandex.direct.web.entity.uac.geoSuggestTextTooLongResponse
import ru.yandex.direct.web.entity.uac.service.UacGeoSuggestService
import ru.yandex.direct.web.entity.uac.toResponse

@RestController
@Api(tags = ["uac"])
@RequestMapping("/uac", produces = [MediaType.APPLICATION_JSON_VALUE])
open class UacGeoSuggestController(
    private val geoSuggestService: UacGeoSuggestService,
    private val authenticationSource: DirectWebAuthenticationSource,
) {

    companion object {
        private const val DEFAULT_RESULTS_COUNT = 10
        private const val DEFAULT_MAX_SUGGEST_TEXT_LEN = 128
    }

    @ApiOperation(
        value = "geoSuggest",
        httpMethod = "GET",
        nickname = "geoSuggest"
    )
    @PreAuthorizeRead
    @GetMapping(value = ["geo_suggest"])
    open fun getGeoSuggest(
        @ApiParam(value = "text", required = true) @RequestParam(required = true) text: String,
        @ApiParam(value = "lang", required = false) @RequestParam(name = "lang", required = false) langParam: String?,
    ): ResponseEntity<Any> {
        if (text.length > DEFAULT_MAX_SUGGEST_TEXT_LEN) {
            return geoSuggestTextTooLongResponse()
        }
        val lang =
            if (langParam == null) authenticationSource.authentication.subjectUser.lang
            else Language.fromLangString(langParam)
        val regions = geoSuggestService.findAvailableRegionsByText(text, lang, DEFAULT_RESULTS_COUNT)
        return ResponseEntity.ok(regions.toResponse())
    }
}
