package ru.yandex.direct.web.entity.uac.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import org.slf4j.LoggerFactory
import org.springframework.http.HttpStatus
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PatchMapping
import org.springframework.web.bind.annotation.PathVariable
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RestController
import org.springframework.web.multipart.MultipartFile
import ru.yandex.direct.audience.client.model.SegmentStatus
import ru.yandex.direct.common.util.HttpUtil
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.feature.service.FeatureHelper.Companion.feature
import ru.yandex.direct.core.entity.metrika.repository.LalSegmentRepository
import ru.yandex.direct.core.entity.mobilegoals.MobileAppGoalsService
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.GoalType
import ru.yandex.direct.core.entity.retargeting.model.Rule
import ru.yandex.direct.core.entity.retargeting.model.RuleType
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.isValidId
import ru.yandex.direct.core.entity.uac.model.UacRetargetingCondition
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.service.AdGroupBriefService
import ru.yandex.direct.core.entity.uac.service.AudienceSegmentsService
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.core.entity.uac.service.RmpCampaignService
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.result.Result
import ru.yandex.direct.web.core.model.retargeting.WebGoalType
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider
import ru.yandex.direct.web.entity.retargetinglists.service.GetRetargetingGoalsService
import ru.yandex.direct.web.entity.uac.converter.UacRetargetingConverter.toUacMobileGoal
import ru.yandex.direct.web.entity.uac.converter.UacRetargetingConverter.toUacRetargetingCondition
import ru.yandex.direct.web.entity.uac.model.RetargetingConditionRequest
import ru.yandex.direct.web.entity.uac.model.UacRetargetingCampaign
import ru.yandex.direct.web.entity.uac.model.UacRetargetingGoals
import ru.yandex.direct.web.entity.uac.notFoundResponse
import ru.yandex.direct.web.entity.uac.service.UacMobileAppService
import ru.yandex.direct.web.entity.uac.service.UacRetargetingConditionAddAndUpdateService
import ru.yandex.direct.web.entity.uac.service.UacRetargetingConditionAddAndUpdateService.Companion.MAX_EXCLUDE_RULES
import ru.yandex.direct.web.entity.uac.service.UacRetargetingConditionAddAndUpdateService.Companion.MAX_INCLUDE_RULES
import ru.yandex.direct.web.entity.uac.service.UacRetargetingConditionAddAndUpdateService.Companion.MIN_INCLUDE_RULES
import ru.yandex.direct.web.entity.uac.service.UacRetargetingConditionAddAndUpdateService.Companion.RETARGETING_CONDITION_CONVERTER_PROVIDER
import ru.yandex.direct.web.entity.uac.service.UacUploadAudienceSegmentFileService
import ru.yandex.direct.web.entity.uac.toResponse
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService

@RestController
@Api(tags = ["uac"])
@RequestMapping("/uac", produces = [MediaType.APPLICATION_JSON_VALUE])
class UacRetargetingController(
    private val authenticationSource: DirectWebAuthenticationSource,
    private val uacMobileAppService: UacMobileAppService,
    private val rmpCampaignService: RmpCampaignService,
    private val mobileAppGoalsService: MobileAppGoalsService,
    private val retargetingConditionService: RetargetingConditionService,
    private val validationResultConversionService: ValidationResultConversionService,
    private val uacRetargetingConditionAddAndUpdateService: UacRetargetingConditionAddAndUpdateService,
    private val campaignService: CampaignService,
    private val grutUacCampaignService: GrutUacCampaignService,
    private val lalSegmentRepository: LalSegmentRepository,
    private val uacUploadAudienceSegmentFileService: UacUploadAudienceSegmentFileService,
    private val getRetargetingGoalsService: GetRetargetingGoalsService,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val audienceSegmentsService: AudienceSegmentsService,
    private val adGroupBriefService: AdGroupBriefService,
) {

    companion object {
        private val logger = LoggerFactory.getLogger(UacRetargetingController::class.java)
    }

    @ApiOperation(
        value = "retargetingGoalsGet",
        httpMethod = "GET",
        nickname = "retargetingGoalsGet",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["retargeting/goals"])
    fun getRetargetingGoals(
        @RequestParam(value = "app_info.id", required = true) appInfoId: String,
        @RequestParam(value = DirectCookieAuthProvider.PARAMETER_ULOGIN) ulogin: String?
    ): ResponseEntity<Any> {
        if (!isValidId(appInfoId)) {
            return notFoundResponse()
        }
        val client = authenticationSource.authentication.subjectUser
        val appInfo = uacMobileAppService.getAppInfo(appInfoId) ?: return notFoundResponse()
        val existingMobileApps = rmpCampaignService.getMobileAppsByUrl(client.clientId, appInfo.url)
        if (existingMobileApps.isEmpty()) {
            return notFoundResponse()
        }
        val existingGoals = mobileAppGoalsService.getGoalsByApps(client.clientId, existingMobileApps)
            .map { toUacMobileGoal(it) }
            .sortedBy { it.goalId }
        return ResponseEntity(
            UacRetargetingGoals(mobileAppId = existingMobileApps.first().id, goals = existingGoals).toResponse(),
            HttpStatus.OK
        )
    }

    @ApiOperation(
        value = "retargetingConditionsGet",
        httpMethod = "GET",
        nickname = "retargetingConditionsGet",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["retargeting/conditions"])
    fun getRetargetingConditions(
        @RequestParam(value = "mobile_app_id", required = false) mobileAppId: Long?,
        @RequestParam(value = DirectCookieAuthProvider.PARAMETER_ULOGIN) ulogin: String?
    ): ResponseEntity<Any> {
        val client = authenticationSource.authentication.subjectUser

        val availableMobileApps =
            if (mobileAppId != null) rmpCampaignService.getMobileAppsByAppId(client.clientId, mobileAppId) else listOf()
        val existingGoalIds = mobileAppGoalsService
            .getGoalsByApps(client.clientId, availableMobileApps)
            .associate { it.id to it.name }

        val existingMetrikaGoalsByGoalTypes = getRetargetingGoalsService
            .getMetrikaGoalsForRetargetingByGoalTypes(setOf(GoalType.AUDIENCE, GoalType.SEGMENT))
        val existingAudienceSegmentIds = existingMetrikaGoalsByGoalTypes
            .filter { it.type == WebGoalType.audience }
            .map { it.id }
            .toSet()
        val existingSegmentIds = existingMetrikaGoalsByGoalTypes
            .filter { it.type == WebGoalType.segment }
            .map { it.id }
            .toSet()

        val lalSegmentIdToParentId = lalSegmentRepository
            .getLalSegmentsByParentIds(existingGoalIds.keys.plus(existingAudienceSegmentIds).plus(existingSegmentIds))
            .associate { it.id to it.parentId }
        val metrikaSegmentGoals = retargetingConditionService
            .getMetrikaGoalsForRetargetingByIds(client.clientId, existingSegmentIds)
            .associateBy { it.id }
        val allExistingIds = existingGoalIds.keys
            .plus(lalSegmentIdToParentId.keys)
            .plus(existingAudienceSegmentIds)
            .plus(existingSegmentIds)

        val rulesFilter = { rules: List<Rule> ->
            rules.isNotEmpty()
                && rules.filter { it.type == RuleType.NOT }.size <= MAX_EXCLUDE_RULES
                && rules.filter {
                it.type in listOf(
                    RuleType.ALL,
                    RuleType.OR
                )
            }.size in MIN_INCLUDE_RULES..MAX_INCLUDE_RULES
        }
        val goalsFilter = { goals: List<Goal> ->
            goals.map(Goal::getId).all(allExistingIds::contains)
        }
        val retargetingConditions = retargetingConditionService.getRetargetingConditions(
            client.clientId,
            null,
            listOf(RetargetingConditionsRetargetingConditionsType.metrika_goals),
            LimitOffset.maxLimited()
        )
            .asSequence()
            .filter { rulesFilter(it.rules) }
            .filter { goalsFilter(it.collectGoals()) }
            .sortedBy { it.id }
            .map { toUacRetargetingCondition(it, existingGoalIds, lalSegmentIdToParentId, metrikaSegmentGoals) }
            .toList()
        return ResponseEntity(retargetingConditions.toResponse(), HttpStatus.OK)
    }

    @ApiOperation(
        value = "retargetingConditionsPost",
        httpMethod = "POST",
        nickname = "retargetingConditionsPost",
    )
    @PreAuthorizeRead
    @PostMapping(value = ["retargeting/conditions"])
    fun createRetargetingCondition(
        @RequestBody request: RetargetingConditionRequest,
        @RequestParam(value = DirectCookieAuthProvider.PARAMETER_ULOGIN) ulogin: String?
    ): ResponseEntity<Any> {
        val client = authenticationSource.authentication.subjectUser

        val result = uacRetargetingConditionAddAndUpdateService.addUacRetargetingCondition(client, request)
        if (!result.isSuccessful) {
            val errors = result.validationResult.flattenErrors().joinToString("; ")
            logger.error("Cannot create retargeting condition: {}", errors)
            return ResponseEntity(
                validationResultConversionService.buildValidationResponse(
                    result.validationResult,
                    RETARGETING_CONDITION_CONVERTER_PROVIDER
                ),
                HttpStatus.BAD_REQUEST,
            )
        }
        return ResponseEntity(listOf(result.result).toResponse(), HttpStatus.OK)
    }

    @ApiOperation(
        value = "campaignsByRetargetingConditions",
        httpMethod = "GET",
        nickname = "campaignsByRetargetingConditions",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["retargeting/campaigns"])
    fun getCampaignsByRetargetingCondition(
        @RequestParam(value = "ret_cond_id", required = true) retargetingConditionId: Long,
        @RequestParam(value = DirectCookieAuthProvider.PARAMETER_ULOGIN) ulogin: String?
    ): ResponseEntity<Any> {
        val operator = authenticationSource.authentication.operator
        val client = authenticationSource.authentication.subjectUser

        val directCampaignIds = retargetingConditionService
            .getCampaignIds(client.clientId, listOf(retargetingConditionId))[retargetingConditionId] ?: listOf()
        val directCampaigns = campaignService.getCampaigns(client.clientId, directCampaignIds)

        val response = directCampaigns.map { UacRetargetingCampaign(name = it.name, id = it.id) }.toMutableList()
        val accountId = grutUacCampaignService.getOrCreateClient(operator, client)
        if (feature(FeatureName.UAC_MULTIPLE_AD_GROUPS_ENABLED).enabled()) {
            val campaignIdsWithAdGroupBriefs =
                adGroupBriefService.getAdGroupBriefsWithRetargetingCondition(
                    accountId, retargetingConditionId
                ).map { it.campaignId }.toSet()
            val campaignNameByIds = grutUacCampaignService.getCampaignNameByIds(campaignIdsWithAdGroupBriefs)
            campaignNameByIds.forEach { response.add(UacRetargetingCampaign(name = it.value, id = it.key)) }
        }
        // todo убрать чтение ретаргетингов из заявки на кампанию https://st.yandex-team.ru/DIRECT-170060
        val uacCampaigns = grutUacCampaignService
            .getCampaignsWithRetargetingCondition(accountId, retargetingConditionId)
        uacCampaigns.forEach { response.add(UacRetargetingCampaign(name = it.name, id = it.id.toIdLong())) }
        return ResponseEntity(response.distinctBy { it.id }.toResponse(), HttpStatus.OK)
    }

    @ApiOperation(
        value = "retargetingConditionsPatch",
        httpMethod = "PATCH",
        nickname = "retargetingConditionsPatch",
    )
    @PreAuthorizeRead
    @PatchMapping(value = ["retargeting/condition/{id}"])
    fun updateRetargetingCondition(
        @PathVariable id: Long,
        @RequestBody request: RetargetingConditionRequest,
        @RequestParam(value = DirectCookieAuthProvider.PARAMETER_ULOGIN) ulogin: String?
    ): ResponseEntity<Any> {
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser

        retargetingConditionService.getRetargetingConditions(
            subjectUser.clientId, listOf(id),
            LimitOffset.maxLimited()
        ).firstOrNull() ?: return notFoundResponse()
        val updateResult = uacRetargetingConditionAddAndUpdateService
            .updateUacRetargetingCondition(subjectUser, request, id)
        if (!updateResult.isSuccessful) {
            val errors = updateResult.validationResult.flattenErrors().joinToString("; ")
            logger.error("Cannot update retargeting condition: {}", errors)
            return ResponseEntity(
                validationResultConversionService.buildValidationResponse(
                    updateResult.validationResult,
                    RETARGETING_CONDITION_CONVERTER_PROVIDER
                ),
                HttpStatus.BAD_REQUEST,
            )
        }

        val accountId = grutUacCampaignService.getOrCreateClient(operator, subjectUser)
        val isCheckAudienceSegmentsDeferredEnabled = feature(FeatureName.CHECK_AUDIENCE_SEGMENTS_DEFERRED).enabled()
        val hasProcessedAudienceSegments =
            isCheckAudienceSegmentsDeferredEnabled && audienceSegmentsService.hasProcessedSegments(
                updateResult.result,
                subjectUser.login
            )
        grutTransactionProvider.runInRetryableTransaction(3) {
            var campaignIdsWithAdGroupBriefAndRetargetingCondition = setOf<Long>()
            if (feature(FeatureName.UAC_MULTIPLE_AD_GROUPS_ENABLED).enabled()) {
                val adGroupBriefs = adGroupBriefService.updateAdGroupBriefsRetargetingCondition(
                    accountId, updateResult.result
                )
                if (adGroupBriefs.isEmpty()) {
                    logger.info("No ad group briefs with retargeting condition id $id")
                } else {
                    val adGroupBriefIds = adGroupBriefs.map { it.id!! }.toSet()
                    campaignIdsWithAdGroupBriefAndRetargetingCondition = adGroupBriefs.map { it.campaignId }.toSet()
                    logger.info(
                        "Retargeting condition $id updated on ad group briefs $adGroupBriefIds " +
                            "from campaigns $campaignIdsWithAdGroupBriefAndRetargetingCondition"
                    )
                    if (hasProcessedAudienceSegments) {
                        campaignIdsWithAdGroupBriefAndRetargetingCondition.forEach {
                            audienceSegmentsService.checkAudienceSegmentsDeferred(
                                subjectUser.clientId, operator.uid, it.toIdString()
                            )
                        }
                    }
                }
            }
            // todo убрать чтение ретаргетингов из заявки на кампанию https://st.yandex-team.ru/DIRECT-170060
            val campaignsToUpdate = grutUacCampaignService.getCampaignsWithRetargetingCondition(accountId, id)
            campaignsToUpdate.forEach {
                grutUacCampaignService.updateCampaign(it.updatedCampaign(updateResult.result))
                if (campaignIdsWithAdGroupBriefAndRetargetingCondition.contains(it.id.toIdLong())) {
                    logger.info("Campaign brief ${it.id} has retargeting condition $id, " +
                        "but ad group briefs already updated")
                } else if (hasProcessedAudienceSegments) {
                    audienceSegmentsService.checkAudienceSegmentsDeferred(subjectUser.clientId, operator.uid, it.id)
                }
            }
        }
        return ResponseEntity(listOf(updateResult.result).toResponse(), HttpStatus.OK)
    }

    @ApiOperation(
        value = "retargetingSegmentsPost",
        httpMethod = "POST",
        nickname = "retargetingSegmentsPost"
    )
    @PreAuthorizeWrite
    @PostMapping(value = ["/retargeting/segments/file"], consumes = [MediaType.MULTIPART_FORM_DATA_VALUE])
    fun uploadAudienceSegments(
        @RequestParam(value = DirectCookieAuthProvider.PARAMETER_ULOGIN) ulogin: String?,
        @RequestParam(value = "file") file: MultipartFile,
        @RequestParam(value = "segmentName") segmentName: String
    ): ResponseEntity<Any> {
        val lang = HttpUtil.getCurrentLocale().map { locale -> locale.language }.orElse(null)
        val subjectUser = authenticationSource.authentication.subjectUser
        val uploadResult = uacUploadAudienceSegmentFileService.uploadAudienceSegmentFile(
            subjectUser,
            file,
            segmentName,
            lang
        )
        if (!uploadResult.isSuccessful) {
            return processError(uploadResult, "File failed to upload: {}")
        }

        return ResponseEntity(uploadResult.result.toResponse(), HttpStatus.OK)
    }

    @ApiOperation(
        value = "retargetingSegments",
        httpMethod = "GET",
        nickname = "retargetingSegments"
    )
    @PreAuthorizeWrite
    @GetMapping(value = ["/retargeting/segments"])
    fun getRetargetingSements(
        @RequestParam(value = DirectCookieAuthProvider.PARAMETER_ULOGIN) ulogin: String?,
        @RequestParam(value = "status") status: SegmentStatus
    ): ResponseEntity<Any> {
        val subjectUser = authenticationSource.authentication.subjectUser
        val uploadResult = uacUploadAudienceSegmentFileService.getRetargetingSegmentsByStatus(
            subjectUser,
            status
        )
        if (!uploadResult.isSuccessful) {
            return processError(uploadResult, "Segments were not obtained: {}")
        }

        return ResponseEntity(uploadResult.result.toResponse(), HttpStatus.OK)
    }

    private fun processError(uploadResult: Result<*>, errorMessage: String): ResponseEntity<Any> {
        val errors = uploadResult.validationResult.flattenErrors().joinToString("; ")
        logger.error(errorMessage, errors)
        return ResponseEntity(
            validationResultConversionService.buildValidationResponse(uploadResult.validationResult),
            HttpStatus.BAD_REQUEST
        )
    }

    private fun UacYdbCampaign.updatedCampaign(
        retargetingCondition: UacRetargetingCondition,
    ): UacYdbCampaign {
        return copy(
            retargetingCondition = retargetingCondition,
        )
    }
}
