package ru.yandex.direct.web.entity.uac.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiParam
import org.slf4j.LoggerFactory
import org.springframework.http.HttpStatus
import org.springframework.http.MediaType.APPLICATION_JSON_VALUE
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.ResponseBody
import org.springframework.web.bind.annotation.RestController
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder
import ru.yandex.direct.core.entity.uac.service.UacDbDefineService
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN
import ru.yandex.direct.web.entity.uac.campaignDoesNotExistResponse
import ru.yandex.direct.web.entity.uac.model.UacAppInfosResponse
import ru.yandex.direct.web.entity.uac.model.UacSiteSuggestResponse
import ru.yandex.direct.web.entity.uac.model.UacSuggestResponse
import ru.yandex.direct.web.entity.uac.notOwnerResponse
import ru.yandex.direct.web.entity.uac.service.UacSuggestService
import ru.yandex.direct.web.entity.uac.toResponse

@RestController
@Api(tags = ["uac"])
@RequestMapping("/uac", produces = [APPLICATION_JSON_VALUE])
open class UacSuggestController(
    private val authenticationSource: DirectWebAuthenticationSource,
    private val uacSuggestService: UacSuggestService,
    private val uacDbDefineService: UacDbDefineService,
    private val uacCampaignServiceHolder: UacCampaignServiceHolder,
    private val grutTransactionProvider: GrutTransactionProvider,
) {
    companion object {
        private val logger = LoggerFactory.getLogger(UacSuggestController::class.java)
        private const val DEFAULT_SUGGEST_SIZE = 10
    }

    private fun getLimit(limit: Int?): Int {
        var currentLimit = limit ?: DEFAULT_SUGGEST_SIZE
        if (currentLimit < 1) {
            currentLimit = DEFAULT_SUGGEST_SIZE
        }
        return currentLimit
    }

    @ApiOperation(
        value = "appSuggestAppsGet",
        httpMethod = "GET",
        nickname = "appSuggestAppsGet",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["suggest"])
    open fun getSuggest(
        @ApiParam(value = "text", required = false) @RequestParam text: String?,
        @ApiParam(value = PARAMETER_ULOGIN, required = false) @RequestParam ulogin: String?,
        @ApiParam(required = false) @RequestParam limit: Int?,
    ): ResponseEntity<UacSuggestResponse> {
        val currentLimit = getLimit(limit)
        val subjectUser = authenticationSource.authentication.subjectUser
        val apps = Trace.current().profile("uacSuggest:suggest").use {
            uacSuggestService.suggest(subjectUser, text, currentLimit)
        }
        return ResponseEntity(apps.toResponse(), HttpStatus.OK)
    }

    @ApiOperation(
        value = "appSuggestLibraryAppsGet",
        httpMethod = "GET",
        nickname = "appSuggestLibraryAppsGet",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["suggest/library"])
    open fun getSuggestLibrary(
        @ApiParam(value = "text", required = false) @RequestParam text: String?,
        @ApiParam(value = PARAMETER_ULOGIN, required = false) @RequestParam ulogin: String?,
        @ApiParam(required = false) @RequestParam limit: Int?,
    ): ResponseEntity<UacAppInfosResponse> {
        val currentLimit = getLimit(limit)
        val subjectUser = authenticationSource.authentication.subjectUser
        val apps = Trace.current().profile("uacSuggest:suggestLibraryApps").use {
            uacSuggestService.suggestLibraryApps(subjectUser, text).take(currentLimit)
        }
        return ResponseEntity(apps.toResponse(), HttpStatus.OK)
    }

    @ApiOperation(
        value = "appSuggestSaasAppsGet",
        httpMethod = "GET",
        nickname = "appSuggestSaasAppsGet",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["suggest/saas"])
    open fun getSuggestSaas(
        @ApiParam(value = "text", required = false) @RequestParam text: String?,
        @ApiParam(value = PARAMETER_ULOGIN, required = false) @RequestParam ulogin: String?,
        @ApiParam(required = false) @RequestParam limit: Int?,
    ): ResponseEntity<UacAppInfosResponse> {
        val currentLimit = getLimit(limit)
        val apps = Trace.current().profile("uacSuggest:suggestSaasApps").use {
            uacSuggestService.suggestSaasApps(text, currentLimit)
        }
        return ResponseEntity(apps.toResponse(), HttpStatus.OK)
    }

    @ApiOperation(
        value = "appSuggestWebUrlsGet",
        httpMethod = "GET",
        nickname = "appSuggestWebUrlsGet",
    )
    @PreAuthorizeRead
    @GetMapping(value = ["suggest/urls"])
    open fun getSuggestWebUrls(
        @ApiParam(value = "text", required = false) @RequestParam text: String?,
        @ApiParam(value = PARAMETER_ULOGIN, required = false) @RequestParam ulogin: String?,
        @ApiParam(required = false) @RequestParam limit: Int?,
    ): ResponseEntity<UacSiteSuggestResponse> {
        val currentLimit = getLimit(limit)
        val subjectUser = authenticationSource.authentication.subjectUser
        val sites = Trace.current().profile("uacSuggest:suggestWebUrls").use {
            uacSuggestService.suggestWebUrls(subjectUser, text).take(currentLimit)
        }
        return ResponseEntity(sites.toResponse(), HttpStatus.OK)
    }

    /**
     * NB! в этой ручке не делается транзакция GrUT потому, что ручка делает запрос в аватарицу, внутри транзакции это делать нельзя
     * Так как сейчас в ручке только один пишущий запрос, то в транщакции смысла нет
     * НО! если будут добавляться еще пишущие запросы, то их нужно объединить в одну транзакцию
     *
     * Ручка используется только для тго
     * Если в саджестах найдены картинки, которые уже есть в кампании (матчим по imageHash или sourceUrl),
     * то возвращаем картинку из кампании со своим content_id, по которому фронт определяет есть ли уже такая картинка
     */
    @ApiOperation(value = "Generate ad images", httpMethod = "GET", nickname = "generateImages")
    @GetMapping(path = ["suggest_images"])
    @ResponseBody
    @PreAuthorizeRead
    open fun getSuggestImages(
        @RequestParam(value = "campaign_id") directCampaignId: Long,
        @RequestParam(value = "ad_group_id", required = false) adGroupId: Long?,
        @RequestParam(required = false) text: String?,
        @RequestParam(required = false) url: String?,
        @RequestParam(value = PARAMETER_ULOGIN, required = false) subjectLogin: String?
    ): ResponseEntity<Any> {
        val operator = authenticationSource.authentication.operator
        val subjectUser = authenticationSource.authentication.subjectUser

        val useGrut = uacDbDefineService.useGrutForDirectCampaignId(directCampaignId)
        uacCampaignServiceHolder.getUacCampaignService(useGrut).getOrCreateClient(operator, subjectUser)
        return getSuggestImagesImpl(useGrut, directCampaignId, operator = operator, subjectUser = subjectUser,
            adGroupId, text, url)

    }

    private fun getSuggestImagesImpl(
        useGrut: Boolean,
        directCampaignId: Long,
        operator: User,
        subjectUser: User,
        adGroupId: Long?,
        text: String?,
        url: String?,
    ): ResponseEntity<Any> {
        val uacCampaignService = uacCampaignServiceHolder.getUacCampaignService(useGrut)
        val uacCampaignId = uacCampaignService.getCampaignIdByDirectCampaignId(directCampaignId)
            ?: return campaignDoesNotExistResponse()
        val uacCampaign = uacCampaignService.getCampaignById(uacCampaignId)
            ?: return campaignDoesNotExistResponse()

        if (!uacCampaignService.checkVisibleCampaign(operator.uid, subjectUser.clientId, directCampaignId)) {
            return notOwnerResponse()
        }

        return uacSuggestService.suggestImages(
            subjectUser,
            operator,
            uacCampaign,
            useGrut,
            directCampaignId,
            adGroupId,
            text,
            url,
        )
    }
}
